#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/json/include/std.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/base64.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>

#include <mapreduce/yt/interface/client.h>

#include <util/system/fs.h>
#include <util/generic/set.h>
#include <library/cpp/string_utils/base64/base64.h>

#include <random>
#include <string>
#include <list>
#include <fstream>

using namespace NYT;

TSet<int64_t> loadFeatureIDs(const TString &path) {
    TSet<int64_t> featureIdsSet;
    std::ifstream ifs(path);
    if (!ifs.is_open())
        return featureIdsSet;
    for (; !ifs.eof();)
    {
        std::string line; std::getline(ifs, line);
        if (line.empty())
            continue;
        if ('#' == line[0])
            continue;
        featureIdsSet.insert(std::stoll(line));
    }
    for (int64_t t : featureIdsSet) {
        INFO() << t;
    }
    return featureIdsSet;
}

void SaveBBox(const TNode& node, maps::json::ObjectBuilder &builder) {
    builder["bbox"] << [&node](maps::json::ArrayBuilder builder) {
        const TVector<TNode>& rcNode = node.AsList();
        builder << [&rcNode](maps::json::ArrayBuilder builder) {
                builder << rcNode[0].AsList()[0].AsInt64();
                builder << rcNode[0].AsList()[1].AsInt64();
            };
        builder << [&rcNode](maps::json::ArrayBuilder builder) {
                builder << rcNode[1].AsList()[0].AsInt64();
                builder << rcNode[1].AsList()[1].AsInt64();
            };
    };
}

void SaveObjects(const TNode& node, maps::json::ObjectBuilder &builder) {
    const TVector<TNode>& objectList = node.AsList();
    builder["objects"] << [&objectList](maps::json::ArrayBuilder builder) {
        for (const TNode& objectNode : objectList) {
            builder << [&objectNode](maps::json::ObjectBuilder builder) {
                builder["type"] = objectNode["type"].AsString();
                builder["num"] = objectNode["num"].AsString();
                SaveBBox(objectNode["bbox"], builder);
            };
        };
    };
}

void Download(IClientPtr client,
              const TString &inputTable,
              const TSet<int64_t> &setFeatureIDs,
              const TString &outputPath) {
    INFO() << "Make result json";

    std::ofstream ofs(outputPath);
    if (!ofs.is_open())
        ERROR() << "Unable to open output file: " << outputPath;
    maps::json::Builder builder(ofs);

    builder << [client, &inputTable, &setFeatureIDs](maps::json::ObjectBuilder builder) {
        builder["images_with_objects"] << [client, &inputTable, &setFeatureIDs](maps::json::ArrayBuilder builder) {
            TTableReaderPtr<TNode> reader = client->CreateTableReader<TNode>(inputTable);
            for (int processedItems = 0; reader->IsValid(); reader->Next(), processedItems++) {
                if ((processedItems + 1) % 1000 == 0) {
                    INFO() << "Processed " << (processedItems + 1) << " items";
                }

                const TNode& inpRow = reader->GetRow();

                int64_t featureID = inpRow["feature_id"].AsInt64();
                if (!setFeatureIDs.empty() && !setFeatureIDs.contains(featureID))
                    continue;
                INFO() << featureID;
                builder << [featureID, &inpRow](maps::json::ObjectBuilder builder) {
                    builder["feature_id"] << featureID;
                    std::stringstream ss;
                    ss << "https://mrc-browser.maps.yandex.net/feature/" << featureID << "/image";
                    builder["source"] << ss.str();
                    builder["image"] << inpRow["image"].AsString();
                    SaveObjects(inpRow["objects"], builder);
                };
            };
        };
    };
}

int main(int argc, const char** argv) try {
    Initialize(argc, argv);

    maps::cmdline::Parser parser("Download traffic signs dataset from YT table to tfrecord");

    maps::cmdline::Option<std::string>
        intputTableName = parser.string("input")
                                .required()
                                .help("Input YT table name");

    maps::cmdline::Option<std::string>
        outputJsonPath = parser.string("output")
                               .required()
                               .help("Path to output json file");

    maps::cmdline::Option<std::string>
        filterFeatureID = parser.string("filter_featureid")
                                .defaultValue("")
                                .help("Path to text file with feature ids, one id on the line");

    parser.parse(argc, const_cast<char**>(argv));

    INFO() << "Connecting to yt::hahn";
    IClientPtr client = CreateClient("hahn");

    TSet<int64_t> featureIdsSet;
    if (filterFeatureID.defined())
        featureIdsSet = loadFeatureIDs(filterFeatureID.c_str());

    Download(client, intputTableName.c_str(), featureIdsSet, outputJsonPath.c_str());
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
