#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/signdetect/include/signdetect_complex.h>

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/json/include/value.h>

#include <library/cpp/string_utils/base64/base64.h>

#include <opencv2/opencv.hpp>

#include <fstream>

namespace {

std::vector<int> loadIds(const std::string& path) {
    std::ifstream ifs(path);
    if (!ifs.is_open())
        return {};

    std::vector<int> result;
    for (; !ifs.eof();) {
        std::string line; std::getline(ifs, line);
        if (line.empty())
           continue;
        result.push_back(std::stoi(line));
    }
    return result;
}

maps::mrc::db::Feature getFeature(maps::pgpool3::Pool& pool, maps::mrc::db::TId featureId) {
    auto txn = pool.slaveTransaction();
    maps::mrc::db::Features features =
        maps::mrc::db::FeatureGateway{*txn}.load(maps::mrc::db::table::Feature::id.equals(featureId));
    REQUIRE(1 == features.size(), "Unable to found feature with id " << featureId);
    return features[0];
}

std::string getFeatureUrl(const maps::mrc::db::Feature& f) {
    return "http://storage-int.mds.yandex.net/get-maps_mrc/" + f.mdsGroupId() + "/" + f.mdsPath();
}

std::vector<uint8_t> downloadImage(maps::http::Client& client, const std::string& url)
{
    maps::common::RetryPolicy retryPolicy;
    retryPolicy.setTryNumber(10)
        .setInitialCooldown(std::chrono::seconds(1))
        .setCooldownBackoff(2);

    auto validateResponse = [](const auto& maybeResponse) {
        return maybeResponse.valid() && maybeResponse.get().responseClass() != maps::http::ResponseClass::ServerError;
    };
    auto resp = maps::common::retry(
                [&]() {
                    return maps::http::Request(client, maps::http::GET, maps::http::URL(url)).perform();
                },
                retryPolicy,
                validateResponse
            );
    REQUIRE(resp.responseClass() == maps::http::ResponseClass::Success,
        "Unexpected response status " << resp.status() << " for url "
        << url);
    return resp.readBodyToVector();
}

cv::Mat loadImage(maps::http::Client& client, const std::string& url, maps::mrc::common::ImageOrientation orientation)
{
    std::vector<uint8_t> data = downloadImage(client, url);
    cv::Mat image = cv::imdecode(data, cv::IMREAD_COLOR | cv::IMREAD_IGNORE_ORIENTATION);
    return maps::mrc::common::transformByImageOrientation(image, orientation);
}

cv::Mat loadFeatureImage(maps::http::Client& client, const maps::mrc::db::Feature& f) {
    return loadImage(client, getFeatureUrl(f), f.orientation());
}

template <typename Detector>
void doDetection(const Detector& tsDetector, const cv::Mat image) {
    maps::mrc::signdetect::DetectedSigns tstSigns = tsDetector.detect(image);
    if (tstSigns.empty()) {
        INFO() << "There are no detections";
        return;
    }
    for (size_t i = 0; i < tstSigns.size(); i++) {
        const maps::mrc::signdetect::DetectedSign& sign = tstSigns[i];
        INFO() << i << ". "
               << maps::mrc::traffic_signs::toString(sign.sign) << " "
               << (sign.number.empty() ? "" : ("number: " + sign.number + " "))
               << ((sign.temporarySign == maps::mrc::traffic_signs::TemporarySign::No) ? "" : "(temporary) ")
               << sign.box;
    }
}

cv::Mat imageDecodeFromBase64(const std::string& src) {
    std::vector<uint8_t> encimage(Base64DecodeBufSize(src.length()), 0);
    encimage.resize(Base64Decode(encimage.data(), src.begin(), src.end()));
    return cv::imdecode(encimage, cv::IMREAD_COLOR + cv::IMREAD_IGNORE_ORIENTATION);
}

void detectOnJson(const std::string& jsonPath, const std::string& jsonOutputPath) {
    maps::mrc::signdetect::SignDetectorComplex tsDetector;
    maps::json::Value fileJson = maps::json::Value::fromFile(jsonPath);
    std::ofstream ofs(jsonOutputPath);
    ofs << "{\n"
        << "    \"images_with_objects\": [\n";

    bool first = true;
    for(const maps::json::Value& item : fileJson["images_with_objects"]) {
        if (first) {
            first = false;
        } else {
            ofs << ",\n";
        }
        std::cout << item["feature_id"].as<int>() << std::endl;
        ofs << "        {\n"
            << "            \"feature_id\": " << item["feature_id"].as<int>() << ",\n"
            << "            \"image\": \"" << item["image"].as<std::string>() << "\",\n"
            << "            \"objects\": [\n";
        bool firstObject = true;
        for (const maps::json::Value& object : item["objects"]) {
            if (firstObject) {
                firstObject = false;
            } else {
                ofs << ",\n";
            }
            ofs << "                {\n"
                << "                    \"type\": \"" << object["type"].as<std::string>() << "\",\n"
                << "                    \"bbox\": [\n"
                << "                        [\n"
                << "                            " << object["bbox"][0][0].as<int>() << ",\n"
                << "                            " << object["bbox"][0][1].as<int>() << "\n"
                << "                        ],\n"
                << "                        [\n"
                << "                            " << object["bbox"][1][0].as<int>() << ",\n"
                << "                            " << object["bbox"][1][1].as<int>() << "\n"
                << "                        ]\n"
                << "                    ]\n"
                << "                }";
        }
        cv::Mat image = imageDecodeFromBase64(item["image"].as<std::string>());
        maps::mrc::signdetect::DetectedSigns tstSigns = tsDetector.detect(image);
        for (size_t i = 0; i < tstSigns.size(); i++) {
            const maps::mrc::signdetect::DetectedSign& sign = tstSigns[i];
            if (firstObject) {
                firstObject = false;
            } else {
                ofs << ",\n";
            }
            ofs << "                {\n"
                << "                    \"type\": \"" << maps::mrc::traffic_signs::toString(sign.sign) << "\",\n";
            if (! sign.number.empty()) {
                ofs << "                    \"num\": \""  << sign.number << "\",\n";
            }
            ofs << "                    \"bbox\": [\n"
                << "                        [\n"
                << "                            " << sign.box.tl().x << ",\n"
                << "                            " << sign.box.tl().y << "\n"
                << "                        ],\n"
                << "                        [\n"
                << "                            " << sign.box.br().x << ",\n"
                << "                            " << sign.box.br().y << "\n"
                << "                        ]\n"
                << "                    ]\n"
                << "                }";
        }
        ofs << "\n"
            << "            ]\n"
            << "        }";
    }
    ofs << "\n"
        << "    ]\n"
        << "}\n";
}

} //namespace

int main(int argc, const char** argv) try {

    maps::cmdline::Parser parser("Launch traffic signs detector-classificator");

    maps::cmdline::Option<std::string> imagePath = parser.string("image")
        .help("Path to image file");

    maps::cmdline::Option<std::string> jsonPath = parser.string("json-input")
        .help("Path to input json file");

    maps::cmdline::Option<std::string> jsonOutputPath = parser.string("json-output")
        .help("Path to output json file");

    maps::cmdline::Option<int> featureId = parser.num("feature-id")
        .help("Feature ID");

    maps::cmdline::Option<std::string> featureIdPath = parser.string("feature-id-list")
        .help("Path to file with list of feature ids");

    maps::cmdline::Option<std::string> mrcConfigPath = parser.string("mrc-config")
        .help("Path to mrc config (needs if feature-id used only)");

    maps::cmdline::Option<std::string> secretVersion = parser.string("secret-version")
        .help("version for secrets from yav.yandex-team.ru (needs if feature-id used only)");

    parser.parse(argc, const_cast<char**>(argv));

    maps::mrc::signdetect::SignDetectorComplex tsDetector;
    //maps::mrc::signdetect::FasterRCNNDetector tsDetector;
    if (imagePath.defined()) {
        doDetection(tsDetector, cv::imread(imagePath));
    } else if (jsonPath.defined() && jsonOutputPath.defined()) {
        detectOnJson(jsonPath, jsonOutputPath);
    } else if (featureId.defined() || featureIdPath.defined()) {
        maps::mrc::common::Config mrcConfig =
            secretVersion.defined()
            ? maps::mrc::common::Config(maps::vault_boy::loadContextWithYaVault(secretVersion), mrcConfigPath)
            : maps::mrc::common::templateConfigFromCmdPath(mrcConfigPath);

        maps::http::Client client;
        maps::wiki::common::PoolHolder mrc(mrcConfig.makePoolHolder());
        std::vector<int> fids;
        if (featureIdPath.defined()) {
            fids = loadIds(featureIdPath);
        } else {
            fids.push_back(featureId);
        }
        for (size_t i = 0; i < fids.size(); i++) {
            INFO() << "FeatureId = " << fids[i];
            doDetection(tsDetector, loadFeatureImage(client, getFeature(mrc.pool(), fids[i])));
        }
    }

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
