#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/base64.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/common/include/retry.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/http/include/http.h>
#include <mapreduce/yt/interface/client.h>
#include <util/system/datetime.h>
#include <util/random/random.h>
#include <library/cpp/string_utils/base64/base64.h>

#include <opencv2/opencv.hpp>

#include <string>
#include <vector>
#include <regex>


static const TString COLUMN_NAME_DATE       = "date";
static const TString COLUMN_NAME_FEATURE_ID = "feature_id";
static const TString COLUMN_NAME_IMAGE      = "image";
static const TString COLUMN_NAME_BASE_IMAGE = "base_image";
static const TString COLUMN_NAME_HOMOGRAPHY = "homography";
static const TString COLUMN_NAME_OBJECTS    = "objects";

static const TString JSON_NAME_IMAGES_SET    = "classification_task_results";
static const TString JSON_NAME_FEATURE_ID    = "feature_id";
static const TString JSON_NAME_BASE_IMAGE    = "base_image";
static const TString JSON_NAME_IMAGE         = "image";
static const TString JSON_NAME_HOMOGRAPHY    = "homography";
static const TString JSON_NAME_VAR_OUT_SIZE  = "var_out_size";
static const TString JSON_NAME_SOURCE_URL    = "source";
static const TString JSON_NAME_OBJECTS       = "solutions";
static const TString JSON_NAME_BBOX          = "bbox";
static const TString JSON_NAME_OBJECT_TYPE   = "sign_type";

std::string download(maps::http::Client& client, const std::string& url) {
    maps::common::RetryPolicy retryPolicy;
    retryPolicy.setTryNumber(10)
        .setInitialCooldown(std::chrono::seconds(1))
        .setCooldownBackoff(2);

    auto validateResponse = [](const auto& maybeResponse) {
        return maybeResponse.valid() && maybeResponse.get().responseClass() != maps::http::ResponseClass::ServerError;
    };
    auto resp = maps::common::retry(
                [&]() {
                    return maps::http::Request(client, maps::http::GET, maps::http::URL(url)).perform();
                },
                retryPolicy,
                validateResponse
            );
    REQUIRE(resp.responseClass() == maps::http::ResponseClass::Success,
        "Unexpected response status " << resp.status() << " for url "
        << url);
    return resp.readBody();
}


NYT::TNode bboxToTNode(const maps::json::Value& bboxJson) {
    return NYT::TNode::CreateList()
        .Add(NYT::TNode::CreateList().Add(bboxJson[0][0].as<int64_t>()).Add(bboxJson[0][1].as<int64_t>()))
        .Add(NYT::TNode::CreateList().Add(bboxJson[1][0].as<int64_t>()).Add(bboxJson[1][1].as<int64_t>()));
}

NYT::TNode objectToTNode(const maps::json::Value& objJson) {
    std::string objType = objJson[JSON_NAME_OBJECT_TYPE].as<std::string>();
    return NYT::TNode()
        ("type", NYT::TNode(objType))
        ("bbox", bboxToTNode(objJson[JSON_NAME_BBOX]));
}

NYT::TNode objectsToTNode(const maps::json::Value& objsJson) {
    NYT::TNode result = NYT::TNode::CreateList();
    for(const maps::json::Value& objJson : objsJson) {
        result.Add(objectToTNode(objJson));
    }
    return result;
}

NYT::TNode homographyToTNode(const maps::json::Value& homJson) {
    return NYT::TNode::CreateList()
        .Add(homJson[0].as<double>())
        .Add(homJson[1].as<double>())
        .Add(homJson[2].as<double>())
        .Add(homJson[3].as<double>())
        .Add(homJson[4].as<double>())
        .Add(homJson[5].as<double>())
        .Add(homJson[6].as<double>())
        .Add(homJson[7].as<double>())
        .Add(homJson[8].as<double>());
}

std::string applyHomography(const std::string &baseImageStr64, const maps::json::Value& homographyJson, const maps::json::Value& varOutSizeJson) {
    TString encBaseImageStr64 = baseImageStr64.c_str();
    std::vector<std::uint8_t> encBaseImage(Base64DecodeBufSize(encBaseImageStr64.length()));
    size_t encBaseImageSize = Base64Decode(encBaseImage.data(), encBaseImageStr64.begin(), encBaseImageStr64.end());
    encBaseImage.resize(encBaseImageSize);

    cv::Mat baseImage = cv::imdecode(encBaseImage, cv::IMREAD_COLOR + cv::IMREAD_IGNORE_ORIENTATION);
    cv::Mat H(3, 3, CV_64FC1);
    for (size_t i = 0; i < 9; i++) {
        H.at<double>(i / 3, i % 3) = homographyJson[i].as<double>();
    }

    cv::Size outSize(varOutSizeJson.as<int>(), varOutSizeJson.as<int>());

    cv::Mat image;
    cv::warpPerspective(baseImage, image, H, outSize);

    std::vector<uint8_t> encImage;
    cv::imencode(".jpg", image, encImage);
    return maps::base64Encode(std::string(encImage.begin(), encImage.end()));
}


NYT::TNode itemToTNode(maps::http::Client& client,
                       const maps::json::Value& itemJson,
                       bool uploadImage)
{
    std::string url = itemJson[JSON_NAME_SOURCE_URL].as<std::string>();
    std::string baseImage = "";
    std::string image = "";
    if (uploadImage) {
        if (itemJson.hasField(JSON_NAME_BASE_IMAGE)) {
            baseImage = itemJson[JSON_NAME_BASE_IMAGE].as<std::string>();
            if (itemJson.hasField(JSON_NAME_IMAGE))
                image = itemJson[JSON_NAME_IMAGE].as<std::string>();
        }
        if (baseImage.empty()) {
            baseImage = maps::base64Encode(download(client, url));
            image = applyHomography(baseImage, itemJson[JSON_NAME_HOMOGRAPHY], itemJson[JSON_NAME_VAR_OUT_SIZE]);
        }
    }

    NYT::TNode result;
    result(COLUMN_NAME_FEATURE_ID, itemJson[JSON_NAME_FEATURE_ID].as<int64_t>())
          (COLUMN_NAME_IMAGE,      NYT::TNode(image))
          (COLUMN_NAME_BASE_IMAGE, NYT::TNode(baseImage))
          (COLUMN_NAME_OBJECTS,    objectsToTNode(itemJson[JSON_NAME_OBJECTS]))
          (COLUMN_NAME_HOMOGRAPHY, homographyToTNode(itemJson[JSON_NAME_HOMOGRAPHY]))
          (COLUMN_NAME_DATE,       "");
    return result;
}

void writeJsonDataset(NYT::TTableWriterPtr<NYT::TNode>& writer,
                     const std::string& path,
                     int limit,
                     bool uploadImages = true)
{
    maps::http::Client client;
    INFO() << "Reading file " << path;
    auto fileJson = maps::json::Value::fromFile(path);

    int processedItems = 0;
    for(const auto& item : fileJson[JSON_NAME_IMAGES_SET])
    try {
        if (limit && processedItems > limit) {
            break;
        }

        writer->AddRow(itemToTNode(client, item, uploadImages));
        ++processedItems;
        if (processedItems % 1000 == 0) {
            INFO() << "Uploaded " << processedItems << " items";
        }
    } catch (const maps::Exception& ex) {
        WARN() << ex;
    }
}

TString temporaryString(const TString& prefix) {
    return prefix + ToString(MicroSeconds()) + "-" + ToString(RandomNumber<ui64>());
}

class TFeatureIDReduce
    : public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>>
{
public:
    Y_SAVELOAD_JOB(UpdateImages_);

    TFeatureIDReduce() = default;

    TFeatureIDReduce(bool updateImages)
        : UpdateImages_(updateImages)
    { }
    void Do(TReader* reader, TWriter* writer) override {
        bool newExists = false;
        NYT::TNode outRow;
        for (; reader->IsValid(); reader->Next()) {
            const auto& curRow = reader->GetRow();

            auto tableIndex = reader->GetTableIndex();
            if (tableIndex == 0) {
                if (newExists) {
                    if (!UpdateImages_)
                        outRow[COLUMN_NAME_IMAGE] = curRow[COLUMN_NAME_IMAGE];
                } else
                    outRow = curRow;
            } else if (tableIndex == 1) {
                outRow[COLUMN_NAME_DATE]       = curRow[COLUMN_NAME_DATE];
                outRow[COLUMN_NAME_FEATURE_ID] = curRow[COLUMN_NAME_FEATURE_ID];
                outRow[COLUMN_NAME_HOMOGRAPHY] = curRow[COLUMN_NAME_HOMOGRAPHY];
                outRow[COLUMN_NAME_OBJECTS]    = curRow[COLUMN_NAME_OBJECTS];
                if (UpdateImages_ || outRow[COLUMN_NAME_BASE_IMAGE].IsUndefined()) {
                    outRow[COLUMN_NAME_BASE_IMAGE] = curRow[COLUMN_NAME_BASE_IMAGE];
                    outRow[COLUMN_NAME_IMAGE] = curRow[COLUMN_NAME_IMAGE];
                }
                newExists = true;
            } else {
                Y_FAIL();
            }
        }
        writer->AddRow(outRow);
    }
private:
    bool UpdateImages_;
};
REGISTER_REDUCER(TFeatureIDReduce)

NYT::TNode createSchema() {
    return NYT::TNode::CreateList()
                .Add(NYT::TNode()("name", COLUMN_NAME_DATE)("type", "string"))
                .Add(NYT::TNode()("name", COLUMN_NAME_FEATURE_ID)("type", "int64"))
                .Add(NYT::TNode()("name", COLUMN_NAME_IMAGE)("type", "string"))
                .Add(NYT::TNode()("name", COLUMN_NAME_BASE_IMAGE)("type", "string"))
                .Add(NYT::TNode()("name", COLUMN_NAME_HOMOGRAPHY)("type", "any"))
                .Add(NYT::TNode()("name", COLUMN_NAME_OBJECTS)("type", "any"));
}


int main(int argc, const char** argv) try {
    const TString YT_TEMPORARY_FOLDER  = "//tmp/maps_mrc";

    NYT::Initialize(argc, argv);

    maps::cmdline::Parser parser("Uploads road marks dataset from file to YT table");

    auto intputFileName = parser.string("input")
        .required()
        .help("Input file path");

    auto outputTableName = parser.string("output")
        .required()
        .help("Output YT table name");

    auto append = parser.flag("append")
        .help("append data to table (by default table rewrited)");

    auto update = parser.flag("update")
        .help("update data into table change rows with same feature_id");

    auto updateImages = parser.flag("update-images")
        .help("applicable only if update flag set, if update-images is set,"\
              "than photo will be redownloading and upload to YT table,"
              "else the photo from current record in YT will be saved");

    auto limit = parser.num("limit");

    parser.parse(argc, const_cast<char**>(argv));

    if (!update && updateImages) {
        WARN() << "'update-images' flag will be ignored, because 'update' flag didn't set";
    }


    INFO() << "Connecting to yt::hahn";
    auto client = NYT::CreateClient("hahn");

    INFO() << "Opening table " << outputTableName;
    if (update) {
        client->Create(YT_TEMPORARY_FOLDER, NYT::NT_MAP, NYT::TCreateOptions().Recursive(true).IgnoreExisting(true));

        TString tmpTable = temporaryString(YT_TEMPORARY_FOLDER + "/");
        client->Create(tmpTable,
                       NYT::NT_TABLE,
                       NYT::TCreateOptions().Attributes(NYT::TNode()("schema", createSchema())));
        //NYT::TTableWriterPtr<NYT::TNode>
        auto writer = client->CreateTableWriter<NYT::TNode>(tmpTable);
        writeJsonDataset(writer, intputFileName, limit, updateImages);
        writer->Finish();

        client->Sort(
            NYT::TSortOperationSpec()
            .AddInput(tmpTable)
            .Output(tmpTable)
            .SortBy({COLUMN_NAME_FEATURE_ID}));

        client->Reduce(
            NYT::TReduceOperationSpec()
                .ReduceBy({COLUMN_NAME_FEATURE_ID})
                .AddInput<NYT::TNode>(outputTableName.c_str())
                .AddInput<NYT::TNode>(tmpTable)
                .AddOutput<NYT::TNode>(outputTableName.c_str()),
            new TFeatureIDReduce(updateImages));
        client->Remove(tmpTable);
    } else {
        if (!append) {
            client->Create(outputTableName.c_str(),
                           NYT::NT_TABLE,
                           NYT::TCreateOptions().Attributes(NYT::TNode()("schema", createSchema())));
        }
        NYT::TRichYPath outRPath = NYT::TRichYPath(outputTableName.c_str()).Append(append);
        auto writer = client->CreateTableWriter<NYT::TNode>(outRPath);
        writeJsonDataset(writer, intputFileName, limit);
        writer->Finish();
    }
    client->Sort(
        NYT::TSortOperationSpec()
        .AddInput(outputTableName.c_str())
        .Output(outputTableName.c_str())
        .SortBy({COLUMN_NAME_FEATURE_ID}));
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
