#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/base64.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/common/include/retry.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/http/include/http.h>
#include <yandex/maps/mrc/traffic_signs/signs.h>
#include <maps/wikimap/mapspro/services/mrc/libs/toloka_manager/conversion.h>
#include <mapreduce/yt/interface/client.h>
#include <util/system/datetime.h>
#include <util/random/random.h>

#include <string>
#include <vector>
#include <regex>


static const TString COLUMN_NAME_DATE       = "date";
static const TString COLUMN_NAME_FEATURE_ID = "feature_id";
static const TString COLUMN_NAME_HEADING    = "heading";
static const TString COLUMN_NAME_IMAGE      = "image";
static const TString COLUMN_NAME_LAT        = "lat";
static const TString COLUMN_NAME_LON        = "lon";
static const TString COLUMN_NAME_OBJECTS    = "objects";

static const std::string JSON_ANSWER_OK     = "ok";

static const TString JSON_NAME_IMAGES_SET   = "images_with_objects";
static const TString JSON_NAME_FEATURE_ID   = "feature_id";
static const TString JSON_NAME_SOURCE_URL   = "source";
static const TString JSON_NAME_IMAGE        = "image";
static const TString JSON_NAME_OBJECTS      = "objects";
static const TString JSON_NAME_ANSWER       = "answer";
static const TString JSON_NAME_BBOX         = "bbox";
static const TString JSON_NAME_OBJECT_TYPE  = "type";
static const TString JSON_NAME_OBJECT_NUM   = "num";
static const TString JSON_NAME_LAT          = "lat";
static const TString JSON_NAME_LON          = "lon";

int64_t extractFeatureIdFromUrl(const std::string& url) {
    std::regex urlRegex("https://mrc-browser.maps.yandex.net/feature/(\\d+)/[\\w]+");
    std::smatch match;
    std::regex_match(url, match, urlRegex);
    REQUIRE(match.size() == 2, "Can't parse url " << url);
    return std::stoll(match[1].str());
}

std::string download(maps::http::Client& client, const std::string& url) {
    maps::common::RetryPolicy retryPolicy;
    retryPolicy.setTryNumber(10)
        .setInitialCooldown(std::chrono::seconds(1))
        .setCooldownBackoff(2);

    auto validateResponse = [](const auto& maybeResponse) {
        return maybeResponse.valid() && maybeResponse.get().responseClass() != maps::http::ResponseClass::ServerError;
    };
    auto resp = maps::common::retry(
                [&]() {
                    return maps::http::Request(client, maps::http::GET, maps::http::URL(url)).perform();
                },
                retryPolicy,
                validateResponse
            );
    REQUIRE(resp.responseClass() == maps::http::ResponseClass::Success,
        "Unexpected response status " << resp.status() << " for url "
        << url);
    return resp.readBody();
}


NYT::TNode bboxToTNode(const maps::json::Value& bboxJson) {
    return NYT::TNode::CreateList()
        .Add(NYT::TNode::CreateList().Add(bboxJson[0][0].as<int64_t>()).Add(bboxJson[0][1].as<int64_t>()))
        .Add(NYT::TNode::CreateList().Add(bboxJson[1][0].as<int64_t>()).Add(bboxJson[1][1].as<int64_t>()));
}

NYT::TNode objectToTNode(const maps::json::Value& objJson) {
    std::string objType = objJson[JSON_NAME_OBJECT_TYPE].as<std::string>();
    std::string objNum;
    if (objJson.hasField(JSON_NAME_OBJECT_NUM) && objJson[JSON_NAME_OBJECT_NUM].isString()) {
        objNum = objJson[JSON_NAME_OBJECT_NUM].as<std::string>();
    }
    return NYT::TNode()
        ("type", NYT::TNode(objType))
        ("num", NYT::TNode(objNum))
        ("bbox", bboxToTNode(objJson[JSON_NAME_BBOX]));
}

NYT::TNode objectsToTNode(const maps::json::Value& objsJson) {
    NYT::TNode result = NYT::TNode::CreateList();
    for(const auto& objJson : objsJson) {
        if (objJson.hasField(JSON_NAME_ANSWER) &&
           (objJson[JSON_NAME_ANSWER].as<std::string>() != JSON_ANSWER_OK)) {
            continue;
        }
        result.Add(objectToTNode(objJson));
    }
    return result;
}

NYT::TNode itemToTNode(maps::http::Client& client,
                       const maps::json::Value& itemJson,
                       bool uploadImage)
{
    const std::string url = itemJson.hasField(JSON_NAME_SOURCE_URL) ? itemJson[JSON_NAME_SOURCE_URL].as<std::string>() : "";
    const double lat = itemJson.hasField(JSON_NAME_LAT) ? itemJson[JSON_NAME_LAT].as<double>() : 0.0;
    const double lon = itemJson.hasField(JSON_NAME_LON) ? itemJson[JSON_NAME_LON].as<double>() : 0.0;

    std::string image = "";
    if (uploadImage) {
        if (itemJson.hasField(JSON_NAME_IMAGE))
            image = itemJson[JSON_NAME_IMAGE].as<std::string>();
        if (image.empty())
            image = maps::base64Encode(download(client, url));
    }
    NYT::TNode result;
    result(COLUMN_NAME_IMAGE,      NYT::TNode(image))
          (COLUMN_NAME_OBJECTS,    objectsToTNode(itemJson[JSON_NAME_OBJECTS]))
          (COLUMN_NAME_FEATURE_ID, itemJson.hasField(JSON_NAME_FEATURE_ID) ? itemJson[JSON_NAME_FEATURE_ID].as<int64_t>() : extractFeatureIdFromUrl(url))
          (COLUMN_NAME_LON,        lon)
          (COLUMN_NAME_LAT,        lat)
          (COLUMN_NAME_HEADING,    0.)
          (COLUMN_NAME_DATE,       "");
    return result;
}

void writeJsonDataset(NYT::TTableWriterPtr<NYT::TNode>& writer,
                     const std::string& path,
                     int limit,
                     bool uploadImages = true)
{
    maps::http::Client client;
    INFO() << "Reading file " << path;
    auto fileJson = maps::json::Value::fromFile(path);

    int processedItems = 0;
    for(const auto& item : fileJson[JSON_NAME_IMAGES_SET])
    try {
        if (limit && processedItems > limit) {
            break;
        }
        writer->AddRow(itemToTNode(client, item, uploadImages));
        ++processedItems;
        if (processedItems % 1000 == 0) {
            INFO() << "Uploaded " << processedItems << " items";
        }
    } catch (const maps::Exception& ex) {
        WARN() << ex;
    }
}

TString temporaryString(const TString& prefix) {
    return prefix + ToString(MicroSeconds()) + "-" + ToString(RandomNumber<ui64>());
}

class TFeatureIDReduce
    : public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>>
{
public:
    Y_SAVELOAD_JOB(UpdateImages_);

    TFeatureIDReduce() = default;

    TFeatureIDReduce(bool updateImages)
        : UpdateImages_(updateImages)
    { }
    void Do(TReader* reader, TWriter* writer) override {
        bool newExists = false;
        NYT::TNode outRow;
        for (; reader->IsValid(); reader->Next()) {
            const auto& curRow = reader->GetRow();

            auto tableIndex = reader->GetTableIndex();
            if (tableIndex == 0) {
                if (newExists) {
                    if (!UpdateImages_)
                        outRow[COLUMN_NAME_IMAGE] = curRow[COLUMN_NAME_IMAGE];
                } else
                    outRow = curRow;
            } else if (tableIndex == 1) {
                outRow[COLUMN_NAME_DATE]       = curRow[COLUMN_NAME_DATE];
                outRow[COLUMN_NAME_FEATURE_ID] = curRow[COLUMN_NAME_FEATURE_ID];
                outRow[COLUMN_NAME_HEADING]    = curRow[COLUMN_NAME_HEADING];
                outRow[COLUMN_NAME_LAT]        = curRow[COLUMN_NAME_LAT];
                outRow[COLUMN_NAME_LON]        = curRow[COLUMN_NAME_LON];
                outRow[COLUMN_NAME_OBJECTS]    = curRow[COLUMN_NAME_OBJECTS];
                if (UpdateImages_ || outRow[COLUMN_NAME_IMAGE].IsUndefined())
                    outRow[COLUMN_NAME_IMAGE] = curRow[COLUMN_NAME_IMAGE];
                newExists = true;
            } else {
                Y_FAIL();
            }
        }
        writer->AddRow(outRow);
    }
private:
    bool UpdateImages_;
};
REGISTER_REDUCER(TFeatureIDReduce)

NYT::TNode createSchema() {
    return NYT::TNode::CreateList()
                .Add(NYT::TNode()("name", COLUMN_NAME_DATE)("type", "string"))
                .Add(NYT::TNode()("name", COLUMN_NAME_FEATURE_ID)("type", "int64"))
                .Add(NYT::TNode()("name", COLUMN_NAME_HEADING)("type", "double"))
                .Add(NYT::TNode()("name", COLUMN_NAME_IMAGE)("type", "string"))
                .Add(NYT::TNode()("name", COLUMN_NAME_LAT)("type", "double"))
                .Add(NYT::TNode()("name", COLUMN_NAME_LON)("type", "double"))
                .Add(NYT::TNode()("name", COLUMN_NAME_OBJECTS)("type", "any"));
}


int main(int argc, const char** argv) try {
    const TString YT_TEMPORARY_FOLDER  = "//tmp/maps_mrc";

    NYT::Initialize(argc, argv);

    maps::cmdline::Parser parser("Uploads traffic signs dataset from file to YT table");

    auto intputFileName = parser.string("input")
        .required()
        .help("Input file path");

    auto outputTableName = parser.string("output")
        .required()
        .help("Output YT table name");

    auto append = parser.flag("append")
        .help("append data to table (by default table rewrited)");

    auto update = parser.flag("update")
        .help("update data into table change rows with same feature_id");

    auto updateImages = parser.flag("update-images")
        .help("applicable only if update flag set, if update-images is set,"\
              "than photo will be redownloading and upload to YT table,"
              "else the photo from current record in YT will be saved");

    auto limit = parser.num("limit");

    parser.parse(argc, const_cast<char**>(argv));

    if (!update && updateImages) {
        WARN() << "'update-images' flag will be ignored, because 'update' flag didn't set";
    }


    INFO() << "Connecting to yt::hahn";
    auto client = NYT::CreateClient("hahn");

    INFO() << "Opening table " << outputTableName;
    if (update) {
        client->Create(YT_TEMPORARY_FOLDER, NYT::NT_MAP, NYT::TCreateOptions().Recursive(true).IgnoreExisting(true));

        TString tmpTable = temporaryString(YT_TEMPORARY_FOLDER + "/");
        client->Create(tmpTable,
                       NYT::NT_TABLE,
                       NYT::TCreateOptions().Attributes(NYT::TNode()("schema", createSchema())));
        auto writer = client->CreateTableWriter<NYT::TNode>(tmpTable);
        writeJsonDataset(writer, intputFileName, limit, updateImages);
        writer->Finish();

        client->Sort(
            NYT::TSortOperationSpec()
            .AddInput(tmpTable)
            .Output(tmpTable)
            .SortBy({COLUMN_NAME_FEATURE_ID}));

        client->Reduce(
            NYT::TReduceOperationSpec()
                .ReduceBy({COLUMN_NAME_FEATURE_ID})
                .AddInput<NYT::TNode>(outputTableName.c_str())
                .AddInput<NYT::TNode>(tmpTable)
                .AddOutput<NYT::TNode>(outputTableName.c_str()),
            new TFeatureIDReduce(updateImages));
        client->Remove(tmpTable);
    } else {
        if (append) {
            client->Create(YT_TEMPORARY_FOLDER, NYT::NT_MAP, NYT::TCreateOptions().Recursive(true).IgnoreExisting(true));

            TString tmpTable = temporaryString(YT_TEMPORARY_FOLDER + "/");
            client->Create(tmpTable,
                NYT::NT_TABLE,
                NYT::TCreateOptions().Attributes(NYT::TNode()("schema", createSchema())));
            auto writer = client->CreateTableWriter<NYT::TNode>(tmpTable);
            writeJsonDataset(writer, intputFileName, limit);
            writer->Finish();
            client->Sort(
                NYT::TSortOperationSpec()
                .AddInput(tmpTable.c_str())
                .Output(tmpTable.c_str())
                .SortBy({ COLUMN_NAME_FEATURE_ID }));

            client->Merge(
                NYT::TMergeOperationSpec()
                .AddInput(tmpTable.c_str())
                .AddInput(outputTableName.c_str())
                .Output(outputTableName.c_str())
                .Mode(NYT::EMergeMode::MM_SORTED));
            client->Remove(tmpTable);
        } else {
            client->Create(outputTableName.c_str(),
                NYT::NT_TABLE,
                NYT::TCreateOptions().Attributes(NYT::TNode()("schema", createSchema())));
            NYT::TRichYPath outRPath = NYT::TRichYPath(outputTableName.c_str());
            auto writer = client->CreateTableWriter<NYT::TNode>(outRPath);
            writeJsonDataset(writer, intputFileName, limit);
            writer->Finish();
            client->Sort(
                NYT::TSortOperationSpec()
                .AddInput(outputTableName.c_str())
                .Output(outputTableName.c_str())
                .SortBy({ COLUMN_NAME_FEATURE_ID }));
        }
    }
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
