#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/base64.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <yandex/maps/mrc/traffic_signs/signs.h>
#include <mapreduce/yt/interface/client.h>

#include <string>
#include <utility>
#include <vector>
#include <map>

namespace ts = maps::mrc::traffic_signs;

using TrafficSigns = std::vector<ts::TrafficSign>;

const std::vector<TrafficSigns>
    TRAFFIC_SIGNS_BY_PRIORITY {
        {
            ts::TrafficSign::ProhibitoryNoEntry,
            ts::TrafficSign::ProhibitoryNoVehicles,
            ts::TrafficSign::ProhibitoryNoHeavyGoodsVehicles,
            ts::TrafficSign::ProhibitoryNoRightTurn,
            ts::TrafficSign::ProhibitoryNoLeftTurn,
            ts::TrafficSign::ProhibitoryNoUturn,
            ts::TrafficSign::ProhibitoryMaxSpeed,
            ts::TrafficSign::ProhibitoryMaxSpeed5,
            ts::TrafficSign::ProhibitoryMaxSpeed10,
            ts::TrafficSign::ProhibitoryMaxSpeed15,
            ts::TrafficSign::ProhibitoryMaxSpeed20,
            ts::TrafficSign::ProhibitoryMaxSpeed25,
            ts::TrafficSign::ProhibitoryMaxSpeed30,
            ts::TrafficSign::ProhibitoryMaxSpeed35,
            ts::TrafficSign::ProhibitoryMaxSpeed40,
            ts::TrafficSign::ProhibitoryMaxSpeed45,
            ts::TrafficSign::ProhibitoryMaxSpeed50,
            ts::TrafficSign::ProhibitoryMaxSpeed55,
            ts::TrafficSign::ProhibitoryMaxSpeed60,
            ts::TrafficSign::ProhibitoryMaxSpeed70,
            ts::TrafficSign::ProhibitoryMaxSpeed80,
            ts::TrafficSign::ProhibitoryMaxSpeed90,
            ts::TrafficSign::ProhibitoryMaxSpeed100,
            ts::TrafficSign::ProhibitoryMaxSpeed110,
            ts::TrafficSign::ProhibitoryMaxSpeed120,
            ts::TrafficSign::ProhibitoryMaxSpeed130,
            ts::TrafficSign::ProhibitoryNoParking,
            ts::TrafficSign::ProhibitoryNoParkingOrStopping,
            ts::TrafficSign::MandatoryProceedStraight,
            ts::TrafficSign::MandatoryProceedStraightOrTurnLeft,
            ts::TrafficSign::MandatoryProceedStraightOrTurnRight,
            ts::TrafficSign::MandatoryTurnLeftAhead,
            ts::TrafficSign::MandatoryTurnRightAhead,
            ts::TrafficSign::PrescriptionOneWayRoad,
            ts::TrafficSign::PrescriptionEofOneWayRoad,
            ts::TrafficSign::PrescriptionEntryToOneWayRoadOnTheRight,
            ts::TrafficSign::PrescriptionEntryToOneWayRoadOnTheLeft,
            ts::TrafficSign::PrescriptionLanesDirection,
            ts::TrafficSign::PrescriptionLaneDirectionFL,
            ts::TrafficSign::PrescriptionLaneDirectionFR,
            ts::TrafficSign::PrescriptionLaneDirectionF,
            ts::TrafficSign::PrescriptionLaneDirectionFl,
            ts::TrafficSign::PrescriptionLaneDirectionFr,
            ts::TrafficSign::PrescriptionLaneDirectionL,
            ts::TrafficSign::PrescriptionLaneDirectionRL,
            ts::TrafficSign::PrescriptionLaneDirectionR,
            ts::TrafficSign::PrescriptionLaneDirectionOther,
            ts::TrafficSign::PrescriptionStartNewLineRight,
            ts::TrafficSign::PrescriptionStartNewLineLeft,
            ts::TrafficSign::PrescriptionEndLineRight,
            ts::TrafficSign::PrescriptionEndLineLeft,
            ts::TrafficSign::PrescriptionLaneDirectionFFr,
            ts::TrafficSign::PrescriptionLaneDirectionFFl,
            ts::TrafficSign::PrescriptionLaneDirectionFrR,
            ts::TrafficSign::PrescriptionLaneDirectionFrFl,
            ts::TrafficSign::PrescriptionLaneDirectionLFl,
            ts::TrafficSign::InformationParking,
        },
        {
            ts::TrafficSign::ProhibitoryMaxWeight,
            ts::TrafficSign::ProhibitoryMaxWeightPerAxle,
            ts::TrafficSign::ProhibitoryMaxHeight,
            ts::TrafficSign::ProhibitoryMaxWidth,
            ts::TrafficSign::ProhibitoryEofMaxSpeed,
            ts::TrafficSign::ProhibitoryNoOvertaking,
            ts::TrafficSign::ProhibitoryEofNoOvertaking,
            ts::TrafficSign::MandatoryRoundabout,
            ts::TrafficSign::PrescriptionBusLane,
            ts::TrafficSign::PrescriptionEofBusLane,
            ts::TrafficSign::WarningRoundaboutAhead,
            ts::TrafficSign::WarningRoadNarrowsOnBoth,
            ts::TrafficSign::WarningPedestrianCrossingAhead,
            ts::TrafficSign::PrescriptionBuiltUpArea,
            ts::TrafficSign::PrescriptionEofBuiltUpArea,
            ts::TrafficSign::InformationDistanceStopAhead,
            ts::TrafficSign::InformationDistanceObjectRight,
            ts::TrafficSign::InformationDistanceObjectLeft,
            ts::TrafficSign::InformationStartZone,
            ts::TrafficSign::InformationInZone,
            ts::TrafficSign::InformationHeavyVehicle,
            ts::TrafficSign::InformationLightVehicle,
            ts::TrafficSign::InformationHoliday,
            ts::TrafficSign::InformationWorkingDayTime,
            ts::TrafficSign::InformationPaidServices,
        },
        {
            ts::TrafficSign::PriorityPriorityRoad,
            ts::TrafficSign::PriorityEofPriorityRoad,
            ts::TrafficSign::PriorityGiveWay,
            ts::TrafficSign::PriorityStop,
            ts::TrafficSign::PriorityStopGesture,
            ts::TrafficSign::WarningUnevenRoadAhead,
            ts::TrafficSign::WarningCrossroadsMinorRoadRL,
            ts::TrafficSign::WarningCrossroadsMinorRoadR,
            ts::TrafficSign::WarningCrossroadsMinorRoadL,
            ts::TrafficSign::WarningCrossroadsMinorRoadFr,
            ts::TrafficSign::WarningCrossroadsMinorRoadFl,
            ts::TrafficSign::WarningCrossroadsMinorRoadBr,
            ts::TrafficSign::WarningCrossroadsMinorRoadBl,
            ts::TrafficSign::MandatoryTurnLeft,
            ts::TrafficSign::MandatoryTurnRight,
        }
    };


int main(int argc, const char** argv) try {
    NYT::Initialize(argc, argv);

    maps::cmdline::Parser parser("Uploads traffic signs priorities table to YT");

    auto outputTableName = parser.string("output")
        .defaultValue("//home/maps/core/mrc/datasets/traffic_signs_priorities")
        .help("Output YT table name");

    parser.parse(argc, const_cast<char**>(argv));

    INFO() << "Connecting to yt::hahn";
    auto client = NYT::CreateClient("hahn");

    INFO() << "Opening table " << outputTableName;
    auto writer = client->CreateTableWriter<NYT::TNode>(outputTableName.c_str());

    int priority = 1;
    for(const auto& signs : TRAFFIC_SIGNS_BY_PRIORITY) {
        for (auto sign : signs) {
            writer->AddRow(
                NYT::TNode()
                ("type", NYT::TNode(ts::toString(sign)))
                ("priority", priority)
            );
        }

        ++priority;
    }
    writer->Finish();

}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
