#pragma once

#include <maps/infra/yacare/include/tvm.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/feedback_client/include/client.h>
#include <maps/libs/pgpool/include/pgpool3.h>
#include <maps/tools/grinder/common/include/client.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/ugc_event_logger/include/logger.h>
#include <yandex/maps/mds/mds.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

namespace maps::mrc::ugc_back {

class Configuration;

using ConfigurationPtr = std::shared_ptr<Configuration>;

class Configuration {
public:
    Configuration(const common::Config& config,
                  const grinder::Config& grinderConfig,
                  ugc_event_logger::Logger& ugcEventLogger,
                  std::chrono::seconds mongoRetriesTimeoutAfterShutdown,
                  std::optional<NTvmAuth::TTvmClient> tvmClient)
        : postgres_(config.makePoolHolder())
        , mds_(config.makeMdsClient())
        , mapsCoreNmapsMrcUgcBackUrl_(
              config.externals().mapsCoreNmapsMrcUgcBackUrl())
        , previewSize_{20}
        , chunkSize_{200}
        , grinderConfig_{grinderConfig}
        , grinderLogger_(grinderConfig_, mongoRetriesTimeoutAfterShutdown)
        , grinderClient_(grinderConfig_, grinderLogger_)
        , ugcEventLogger_(ugcEventLogger)
        , contributionsModifyUrl_([&config] {
            auto result = config.externals().mapsCoreUgcBackofficeUrl();
            result.setPath("/v1/contributions/modify");
            return result;
        }())
        , tvmClient_(std::move(tvmClient))
        , feedbackClient_(std::make_unique<feedback_client::Client>(
            makeFeedbackClientConfig(config)))
    {
    }

    static ConfigurationPtr& instance()
    {
        static ConfigurationPtr instance_;
        return instance_;
    }

    static ConfigurationPtr swap(ConfigurationPtr configuration)
    {
        std::swap(instance(), configuration);
        return configuration;
    }

    pgpool3::Pool& pool() { return postgres_.pool(); }

    maps::mds::Mds& mds() { return mds_; }

    const http::URL& mapsCoreNmapsMrcUgcBackUrl() const
    {
        return mapsCoreNmapsMrcUgcBackUrl_;
    }

    size_t previewSize() { return previewSize_; }

    size_t chunkSize() { return chunkSize_; }

    size_t swapPreviewSize(size_t previewSize)
    {
        ASSERT(previewSize > 0);
        return std::exchange(previewSize_, previewSize);
    }

    size_t swapChunkSize(size_t chunkSize)
    {
        ASSERT(chunkSize > 0);
        return std::exchange(chunkSize_, chunkSize);
    }

    grinder::Client& grinderClient() { return grinderClient_; }

    ugc_event_logger::Logger& ugcEventLogger() { return ugcEventLogger_; }

    const http::URL& contributionsModifyUrl() const
    {
        return contributionsModifyUrl_;
    }

    const std::optional<NTvmAuth::TTvmClient>& tvmClient() const
    {
        return tvmClient_;
    }

    feedback_client::IClient& feedbackClient()
    {
        ASSERT(feedbackClient_);
        return *feedbackClient_;
    }

    feedback_client::Configuration
    makeFeedbackClientConfig(const common::Config& mrcConfig)
    {
        auto config = feedback_client::Configuration(
            mrcConfig.externals().socialBackofficeUrl().host());
        if (tvmClient_) {
            config.setTvmTicketProvider([this](){
                return tvmClient_->GetServiceTicketFor("social-backoffice");
            });
        }
        return config;
    }

    /// Used only in tests
    void setFeedbackClient(std::unique_ptr<feedback_client::IClient> client)
    {
        ASSERT(client);
        feedbackClient_ = std::move(client);
    }

private:
    wiki::common::PoolHolder postgres_;
    mds::Mds mds_;
    http::URL mapsCoreNmapsMrcUgcBackUrl_;
    size_t previewSize_;
    size_t chunkSize_;
    grinder::Config grinderConfig_;
    grinder::MongoTaskLogger grinderLogger_;
    grinder::Client grinderClient_;
    ugc_event_logger::Logger& ugcEventLogger_;
    http::URL contributionsModifyUrl_;
    std::optional<NTvmAuth::TTvmClient> tvmClient_;
    std::unique_ptr<feedback_client::IClient> feedbackClient_;
};

inline ConfigurationPtr makeConfiguration(
    const common::Config& config,
    const grinder::Config& grinderConfig,
    ugc_event_logger::Logger& ugcEventLogger,
    std::chrono::seconds mongoRetriesTimeoutAfterShutdown,
    std::optional<NTvmAuth::TTvmClient> tvmClient)
{
    return std::make_shared<Configuration>(config,
                                           grinderConfig,
                                           ugcEventLogger,
                                           mongoRetriesTimeoutAfterShutdown,
                                           std::move(tvmClient));
}

}  // namespace maps::mrc::ugc_back
