#include <maps/wikimap/mapspro/services/rdrproxy/lib/config.h>
#include <maps/wikimap/mapspro/libs/acl_utils/include/caching_acl.h>

#include <maps/infra/yacare/include/yacare.h>
#include <maps/infra/yacare/include/params/tvm.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <maps/libs/auth/include/tvm.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/http/include/http.h>

#include <maps/libs/common/include/environment.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/profiletimer.h>

#include <memory>
#include <string>
#include <sstream>

namespace rdrproxy = maps::wiki::rdrproxy;
namespace http = maps::http;
namespace mw = maps::wiki;


namespace {

const std::string DEFAULT_LAYERS_CONFIG
    = "/etc/yandex/maps/wiki/rdrproxy/layers.json";

const size_t ACL_CACHE_MAX_SIZE = 1000;
const std::chrono::seconds ACL_CACHE_EXPIRATION_PERIOD(600);


std::unique_ptr<const rdrproxy::LayersConfig> g_layersConfig;
std::unique_ptr<maps::wiki::common::ExtendedXmlDoc> g_configHolder;
std::unique_ptr<maps::wiki::common::PoolHolder> g_poolHolder;
std::unique_ptr<mw::acl_utils::CachingAclChecker> g_acl;
std::optional<NTvmAuth::TTvmClient> g_tvmClient;

void errorReporter(const yacare::Request&, yacare::Response& resp)
{
    try {
        throw;
    }
    catch (const yacare::Error& e) {
        resp.setStatus(yacare::HTTPStatus::find(e.status()));
        resp << e.what();
        ERROR() << e;
    }
    catch (const maps::Exception& e) {
        resp.setStatus(yacare::HTTPStatus::InternalError);
        ERROR() << e;
    }
    catch (const std::exception& e) {
        resp.setStatus(yacare::HTTPStatus::InternalError);
        ERROR() << "Internal error: " << e.what();
    }
    catch (...) {
        resp.setStatus(yacare::HTTPStatus::InternalError);
        ERROR() << "unknown error";
    }
}


std::string getServiceTicketFor(const std::string& serviceTvmAlias)
{
    ASSERT(g_tvmClient);
    return g_tvmClient->GetServiceTicketFor(serviceTvmAlias.c_str());
}


http::Response performProxyRequest(http::Client& client,
                                   const rdrproxy::LayerConfig& layerConfig)
{
    static const std::vector<std::string> CACHE_CONTROL_HEADERS{
        "If-None-Match", "If-Modified-Since", "Cache-Control"};

    http::URL url(layerConfig.url);
    for (const auto& param : layerConfig.bypassParams) {
        if (yacare::request().input().has(param)) {
            url.addParam(param, yacare::request().input()[param]);
        }
    }

    http::Request request(client, http::GET, url);

    if (layerConfig.serviceTvmAlias.has_value()) {
        request.addHeader(maps::auth::SERVICE_TICKET_HEADER,
            getServiceTicketFor(layerConfig.serviceTvmAlias.value()));
    }

    for (const auto& header : CACHE_CONTROL_HEADERS) {
        auto headerIt = yacare::request().env().find(header);
        if (headerIt != yacare::request().env().end()) {
            request.addHeader(header, headerIt->second);
        }
    }

    INFO() << "performing proxy request " << url;
    ProfileTimer timer;
    auto response = request.perform();
    INFO() << "the request to " << url << " has been performed in " << timer.getElapsedTime() << " sec";
    return response;
}

void fillYacareResponse(http::Response& r)
{
    ProfileTimer timer;
    yacare::response().setStatus(yacare::HTTPStatus::find(r.status()));
    static const std::vector<std::string> PROXIED_HEADERS{
        "Cache-Control", "ETag", "Last-Modified", "Content-Type", "Content-Length", "Expires"};

    auto requestHeaders = r.headers();
    for (const auto& header : PROXIED_HEADERS) {
        auto headerIt = requestHeaders.find(header);
        if (headerIt != requestHeaders.end()) {
            yacare::response().addHeader(header, headerIt->second);
        }
    }

    if (!r.body().eof()) {
        yacare::response() << r.body().rdbuf();
    }
    INFO() << "the data has been written in " << timer.getElapsedTime() << " sec";
}

void handleLayerRequest(
    const rdrproxy::LayerConfig& layerConfig,
    uint64_t userId)
{
    if (layerConfig.permission.has_value()) {
        if (!g_acl->userHasPermission(userId, layerConfig.permission.value())) {
            WARN() << "user " << userId << " does not have permission" << layerConfig.permission.value();
            throw yacare::errors::Forbidden();
        }
    }

    http::Client client;
    auto layerResponse = performProxyRequest(client, layerConfig);

    if (layerResponse.status() < 500)
    {
        fillYacareResponse(layerResponse);
    } else {
        WARN() << "unexpected return status " << layerResponse.status();
        throw yacare::errors::InternalError() << "failed to perform request";
    }
}

} // namespace

yacare::VirtualHost vhost{
    yacare::VirtualHost::SLB{"core-nmaps-rdrproxy"}
};

yacare::ThreadPool satellitePool("satellite", 15 * yacare::ThreadPool::CPUCOUNT - 1, 512);
yacare::ThreadPool cartographPool("cartograph", 15 * yacare::ThreadPool::CPUCOUNT - 1, 512);

YCR_USE(vhost) {

YCR_RESPOND_TO("/layer/satellite", userId, YCR_IN_POOL(satellitePool))
{
    handleLayerRequest(g_layersConfig->at("satellite"), userId);
}

YCR_RESPOND_TO("/layer/cartograph", userId, YCR_IN_POOL(cartographPool))
{
    handleLayerRequest(g_layersConfig->at("cartograph"), userId);
}

} // YCR_USE

YCR_MAIN(argc, argv)
try {
    maps::cmdline::Parser parser;

    auto configPath = parser.string("config");

    auto layersConfigPath = parser.string("layers-config")
                                .defaultValue(DEFAULT_LAYERS_CONFIG)
                                .help("path to layers configuration");

    auto debug = parser.flag('d', "debug")
                    .help("log debug messages");

    parser.parse(argc, argv);

    auto tvmtoolSettings = maps::auth::TvmtoolSettings();
    g_tvmClient = tvmtoolSettings.makeTvmClient();

    if (debug) {
        maps::log8::setLevel(maps::log8::Level::DEBUG);
    }

    if (configPath.defined()) {
        g_configHolder.reset(
            new maps::wiki::common::ExtendedXmlDoc(configPath));
    }
    else {
        g_configHolder = maps::wiki::common::loadDefaultConfig();
    }

    auto layersConfigJson = maps::json::Value::fromFile(layersConfigPath);

    g_layersConfig.reset(new rdrproxy::LayersConfig(
        rdrproxy::loadConfig(layersConfigJson)));

    g_poolHolder.reset(new maps::wiki::common::PoolHolder(
        *g_configHolder, "core" /* db */, "core" /* pool */));

    g_acl.reset(new mw::acl_utils::CachingAclChecker(
        g_poolHolder->pool(),
        ACL_CACHE_MAX_SIZE,
        ACL_CACHE_EXPIRATION_PERIOD));

    yacare::setErrorReporter(errorReporter);
    if (maps::common::getYandexEnvironment() != maps::common::Environment::Development) {
        INFO() << "Enable TVM suppport";
        yacare::tvm::configureUserAuth(maps::auth::BlackboxApi(tvmtoolSettings));
    }
    yacare::run(yacare::RunSettings{.useSystemDefaultLocale = true});

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    std::cerr << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    std::cerr << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    std::cerr << "Caught unknown exception";
    return EXIT_FAILURE;
}

