#include <maps/wikimap/mapspro/services/renderer/src/data_sets/include/source_layer.h>

#include <maps/renderer/libs/base/include/exception.h>
#include <maps/libs/enum_io/include/enum_io.h>

#include <contrib/libs/yaml-cpp/include/yaml-cpp/yaml.h>

#include <unordered_set>

namespace maps::wiki::renderer {

using namespace mr::data_set;

constexpr maps::enum_io::Representations<FeatureGeneratorType> FEATURE_GENERATOR_TYPE_REPRESENTATION{
    {FeatureGeneratorType::RoadMarkingSymbol,   "road_marking_symbol"},
    {FeatureGeneratorType::RoadMarkingLine,     "road_marking_line"},
    {FeatureGeneratorType::RoadMarkingPolygon,  "road_marking_polygon"}
};
DEFINE_ENUM_IO(FeatureGeneratorType, FEATURE_GENERATOR_TYPE_REPRESENTATION);

constexpr double BBOX_EXTEND_FOR_ROAD_MARKING_SYMBOL = 0.5;

namespace {

std::vector<std::string> getRequiredProperties(
    FeatureGeneratorType type)
{
    switch (type){
        case FeatureGeneratorType::RoadMarkingSymbol:
            return {"ft_type_id", "yaw"};
        case FeatureGeneratorType::RoadMarkingLine:
            return {"ft_type_id"};
        case FeatureGeneratorType::RoadMarkingPolygon:
            return {"ft_type_id"};
    }
}

schema::Property makeProperty(
    schema::PropertyType type,
    std::set<std::string> enumValues)
{
    schema::Property property(type);
    property.enumValues = std::move(enumValues);
    return property;
}

std::map<std::string, schema::Property> getGeneratedProperties(
    FeatureGeneratorType type)
{
    switch (type){
        case FeatureGeneratorType::RoadMarkingSymbol:
            return {};
        case FeatureGeneratorType::RoadMarkingLine:
            return {
                {"width",  makeProperty(schema::PropertyType::String, {"10", "15", "20", "30", "40", "60"})},
                {"dash",   makeProperty(schema::PropertyType::String, {"50_50", "90_30", "100_300", "300_100", "100_100", "40_20"})},
                {"color",  makeProperty(schema::PropertyType::String, {"yellow", "white"})},
                {"side",   makeProperty(schema::PropertyType::String, {"left", "right"})}
            };
        case FeatureGeneratorType::RoadMarkingPolygon:
            return {
                {"color",  makeProperty(schema::PropertyType::String, {"yellow", "white"})}
            };
    }
}

schema::GeometryType geometryType(mr::feature::FeatureType type)
{
    switch (type) {
        case mr::feature::FeatureType::Point:
            return schema::GeometryType::Point;
        case mr::feature::FeatureType::Polyline:
            return schema::GeometryType::Line;
        case mr::feature::FeatureType::Polygon:
            return schema::GeometryType::Area;
        default:
            throw maps::RuntimeError()
                << "Unknown feature type value: " << static_cast<int>(type);
    }
}

std::optional<std::string> getOptString(const YAML::Node& doc, const std::string& name)
{
    const auto node = doc[name];
    if (node.IsDefined()) {
        auto str = node.as<std::string>();
        if (!str.empty())
            return str;
    }
    return std::nullopt;
}

SourceLayer sourceLayerFromYaml(const YAML::Node& doc)
{
    const auto idNode = doc["id"];
    const auto tableNode = doc["table"];
    const auto typeNode = doc["type"];
    REN_REQUIRE(idNode.IsDefined(), "'id' field is required");
    REN_REQUIRE(tableNode.IsDefined(), "'table' field is required");
    REN_REQUIRE(typeNode.IsDefined(), "'type' field is required");

    auto featureType = enum_io::fromString<mr::feature::FeatureType>(
        typeNode.as<std::string>());

    return {
        idNode.as<std::string>(),
        tableNode.as<std::string>(),
        featureType};
}

void fillFromYaml(SourceLayer& layer, const YAML::Node& doc)
{
    const auto idNode = doc["id"];
    REN_REQUIRE(idNode.IsDefined(), "'id' field is required");
    layer.id = idNode.as<std::string>();
    layer.info = LayerInfo{layer.id, layer.featureType};

    const auto geomExpressionNode = doc["geometry_expression"];
    if (geomExpressionNode.IsDefined()) {
        layer.geometryExpression = geomExpressionNode.as<std::string>();
    }

    const auto minZoomNode = doc["minzoom"];
    if (minZoomNode.IsDefined()) {
        layer.minzoom = minZoomNode.as<uint32_t>();
    }

    const auto zlevelNode = doc["zlevel"];
    if (zlevelNode.IsDefined()) {
        REN_REQUIRE(zlevelNode.IsSequence() && zlevelNode.size() == 2,
                    "'zlevel' field must be an array of size 2");

        layer.zlevel = {zlevelNode[0].as<std::string>(),
                        zlevelNode[1].as<std::string>()};
        layer.info.setHasZLevel();
    }

    const auto zoomRangeNode = doc["zoom_range"];
    if (zoomRangeNode.IsDefined()) {
        REN_REQUIRE(zoomRangeNode.IsSequence() && zoomRangeNode.size() == 2,
                    "'zoom_range' field must be an array of size 2");
        layer.zoomRange = {zoomRangeNode[0].as<std::string>(),
                           zoomRangeNode[1].as<std::string>()};
    }

    if (auto res = getOptString(doc, "zoom_expression")) {
        layer.zoomExpression = res.value();
    }

    if (auto res = getOptString(doc, "filter")) {
        layer.filterClause = res.value();
    }

    if (auto res = getOptString(doc, "selection_filter_clause1")) {
        layer.selectionFilterClause1 = res.value();
    }

    if (auto res = getOptString(doc, "selection_filter_clause2")) {
        layer.selectionFilterClause2 = res.value();
    }

    if (auto res = getOptString(doc, "feature_generator")) {
        layer.featureGenerator = enum_io::fromString<FeatureGeneratorType>(res.value());
    }

    if (layer.featureGenerator &&
        layer.featureGenerator.value() == FeatureGeneratorType::RoadMarkingSymbol) {
            layer.info.setTileBboxBuffer(BBOX_EXTEND_FOR_ROAD_MARKING_SYMBOL);
    }

    auto& properties = layer.schema.properties;

    properties.emplace(layer.featureId, schema::PropertyType::Number);
    layer.propertyExpressions.insert_or_assign(layer.featureId, std::nullopt);

    const auto propertiesNode = doc["properties"];
    if (propertiesNode.IsDefined()) {
        REN_REQUIRE(propertiesNode.IsSequence(),
                    "'properties' field must be an array");

        for (const auto& propNode : propertiesNode) {
            auto name = propNode["name"].as<std::string>();
            auto type = enum_io::fromString<schema::PropertyType>(
                propNode["type"].as<std::string>("string"));

            auto res = properties.emplace(name, schema::Property(type));
            REN_REQUIRE(res.second,
                "found duplicate property with name " << name);
            auto& property = res.first->second;
            auto valuesNode = propNode["values"];
            if (valuesNode.IsDefined()) {
                REN_REQUIRE(valuesNode.IsSequence(),
                    "'values' field must be an array");
                REN_REQUIRE(type == schema::PropertyType::String,
                    "'values' must be used for string property");
                std::set<std::string> values;
                for (const auto valueNode : valuesNode) {
                    values.insert(valueNode.as<std::string>());
                }
                property.enumValues = values;
            }
            auto expr = getOptString(propNode, "expression");
            layer.propertyExpressions.emplace(name, expr);
        }
    }

    if (layer.featureGenerator) {
        auto featureGenerator = layer.featureGenerator.value();
        auto required = getRequiredProperties(featureGenerator);
        for (const auto& name : required) {
            REN_REQUIRE(properties.contains(name), "property \"" << name << "\" is required for feature generator");
        }
        auto generated = getGeneratedProperties(featureGenerator);
        for (const auto& [name, property] : generated) {
            REN_REQUIRE(!properties.contains(name), "feature generator "
                << toString(featureGenerator) << " generates predefined "
                << name << " property, it must not be specified in config");
            properties.emplace(name, property);
        }
    }

    std::vector<Property> layerInfoProperties;
    for (const auto& [name, _] : properties) {
        layerInfoProperties.push_back({name, std::nullopt});
    }
    layer.info.setProperties(std::move(layerInfoProperties));
}

class SourceLayerLoader
{
public:
    SourceLayerLoader(const YAML::Node& layersNode)
    {
        for (const auto& layerNode : layersNode) {
            const auto idNode = layerNode["id"];
            REN_REQUIRE(idNode.IsDefined(), "'id' field is required");
            const auto id = idNode.as<std::string>();
            REN_REQUIRE(!layerNodes_.contains(id), "Found duplicated source layer with id='" << id <<"'");
            layerNodes_.insert({id, layerNode});
        }

        for (const auto& [id, layerNode] : layerNodes_) {
            getLayer(id);
        }
    }

    SourceLayers && sourceLayers() {
        return std::move(loaded);
    }

private:
    SourceLayer& getLayer(const std::string& id) {
        if (loaded.find(id) == loaded.end()) {
            auto layer = loadFromYaml(id);
            loaded.insert({id, std::move(layer)});
            return loaded.at(id);
        }
        return loaded.at(id);
    }

    SourceLayer loadFromYaml(const std::string& id) {
        REN_REQUIRE(layerNodes_.contains(id), "Can not find layer with id='" << id << "'");

        REN_REQUIRE(!inProcess.contains(id), "Logic error, layer with id='" << id << "' is allready in process");
        inProcess.insert(id);

        const auto& layerNode = layerNodes_.at(id);

        SourceLayer layer = [&] () {
            const auto extendNode = layerNode["extend"];
            if (extendNode.IsDefined()) {
                auto parentId = extendNode.as<std::string>();
                auto& parent = getLayer(parentId);
                return parent;
            } else {
                return sourceLayerFromYaml(layerNode);
            }
        }();
        fillFromYaml(layer, layerNode);

        inProcess.erase(id);
        return std::move(layer);
    }

private:
    std::unordered_map<std::string, YAML::Node> layerNodes_;
    SourceLayers loaded;
    std::unordered_set<std::string> inProcess;
};

} // namespace

SourceLayer::SourceLayer(
    std::string id_,
    std::string table_,
    mr::feature::FeatureType featureType_)
    : id(std::move(id_))
    , table(std::move(table_))
    , featureType(featureType_)
    , info(id, featureType)
    , schema(geometryType(featureType))
{
}

SourceLayers loadSourceLayers(const YAML::Node& layersNode)
{
    SourceLayerLoader loader(layersNode);
    return std::move(loader.sourceLayers());
}

} // namespace maps::wiki::renderer
