#include <maps/wikimap/mapspro/services/renderer/src/data_sets/include/data_set.h>
#include <maps/wikimap/mapspro/services/renderer/src/data_sets/include/source_layer.h>
#include <maps/wikimap/mapspro/libs/unittest/include/yandex/maps/wiki/unittest/arcadia.h>
#include <library/cpp/testing/gmock_in_unittest/gmock.h>
#include <contrib/libs/yaml-cpp/include/yaml-cpp/yaml.h>

namespace maps::wiki::tests {

namespace mr = maps::renderer;

Y_UNIT_TEST_SUITE(geometry_expression_tests)
{

Y_UNIT_TEST(load_layers_with_custom_geometry_test)
{
    const auto layersYaml = R"(
        - id: layer_line1
          type: polyline
          minzoom: 7
          zoom_expression: 'zmin <= %Z% AND %Z% <= zmax'
          table: |
            (
              SELECT
                id,
                the_geom as geom,
                zmin,
                zmax
              FROM objects_l_view
              WHERE domain_attrs?'cat:rd_el'
            )

        - id: layer_point1
          type: point
          minzoom: 7
          zoom_expression: 'zmin <= %Z% AND %Z% <= zmax'
          geometry_expression: geom_expression
          table: |
            (
              SELECT
                id,
                the_geom as geom,
                ST_ClosestPoint(the_geom,ST_Centroid(the_geom)) as geom_expression,
                zmin,
                zmax
              FROM objects_a_view
              WHERE domain_attrs?'cat:error'
            )

        - id: layer_point2
          extend: layer_point1
    )";
    auto layersNode = YAML::Load(layersYaml);
    auto layers = maps::wiki::renderer::loadSourceLayers(layersNode);

    struct LayersGeometries {
        std::string id;
        std::string geometryColumn;
        std::string geometryExpression;
    };

    auto testLayer = [&](const LayersGeometries& expected) {
        EXPECT_TRUE(layers.contains(expected.id));
        const auto& layer = layers.at(expected.id);
        EXPECT_EQ(layer.id, expected.id);
        EXPECT_EQ(layer.geometryColumn, expected.geometryColumn);
        EXPECT_EQ(layer.geometryExpression, expected.geometryExpression);
    };

    testLayer({
        .id = "layer_line1",
        .geometryColumn = "geom",
        .geometryExpression = "geom"
    });

    testLayer({
        .id = "layer_point1",
        .geometryColumn = "geom",
        .geometryExpression = "geom_expression"
    });

    testLayer({
        .id = "layer_point2",
        .geometryColumn = "geom",
        .geometryExpression = "geom_expression"
    });
}

Y_UNIT_TEST_F(check_geometry_expression_test, unittest::ArcadiaDbFixture)
{
    const auto layersYaml = R"(
        - id: errors_point1
          type: point
          minzoom: 7
          geometry_expression: 'geom_expression'
          table: |
            (
              SELECT
                id,
                the_geom as geom,
                ST_ClosestPoint(the_geom,ST_Centroid(the_geom)) as geom_expression
              FROM objects_a_view
              WHERE domain_attrs?'cat:error'
            )

        - id: errors_point2
          type: point
          minzoom: 7
          geometry_expression: 'ST_ClosestPoint(geom,ST_Centroid(geom))'
          table: |
            (
              SELECT
                id,
                the_geom as geom
              FROM objects_a_view
              WHERE domain_attrs?'cat:error'
            )

        - id: errors_polygon
          type: polygon
          minzoom: 7
          table: |
            (
              SELECT
                id,
                the_geom as geom
              FROM objects_a_view
              WHERE domain_attrs?'cat:error'
            )
    )";

    const auto groupsYaml = R"(
        - id: errors_group
          label: "{{layer-groups:errors_group}}"
          layers:
            - id: errors1
              name: errors
              label: "errors1"
              source_layer:
                - id: errors_point1

            - id: errors2
              name: errors2
              label: "errors2"
              source_layer:
                - id: errors_point2

            - id: errors_polygon
              name: errors_polygon
              label: "errors_polygon"
              source_layer:
                - id: errors_polygon
    )";

    YAML::Node groupsNode = YAML::Load(groupsYaml);
    YAML::Node layersNode = YAML::Load(layersYaml);

    auto datasetHolder = std::make_unique<maps::wiki::renderer::DataSet>(layersNode, groupsNode);

    std::stringstream query;
    query   << "SET search_path=vrevisions_trunk,public; ";
    query   << "INSERT INTO vrevisions_trunk.objects_a("
            << "    id, commit_id, the_geom, zmin, zmax, domain_attrs, service_attrs, area) "
            << "VALUES (1234567890, 1, "
            << "    ST_GeomFromText('MULTIPOLYGON(((25 25, 75 25, 75 75, 25 75, 25 25)))', 3395), "
            << "    11, 21, hstore(ARRAY['cat:error', '1', 'error:type', '10', 'error:status', '6']), null, 2500);";
    database().executeSql(query.str());

    maps::wiki::renderer::AuxData auxData;
    auxData.searchPath = "vrevisions_trunk,public";
    auxData.getTransaction = [&]() {
        auto connection = pool().getSlaveConnection("");
        return pgpool3::makeNonTransaction(std::move(connection));
    };

    mr::base::Zoom z = 21;
    mr::data_set::ViewQueryParams params{mr::base::BoxD{0, 0, 100, 100}, mr::base::ZoomRange{z, z}};
    params.auxData = auxData;

    mr::base::Vertices expectedPoint;
    expectedPoint.addGeomPoint({50, 50});

    mr::base::Vertices expectedPolygon;
    expectedPolygon.addMoveTo({25, 25});
    expectedPolygon.addLineTo({75, 25});
    expectedPolygon.addLineTo({75, 75});
    expectedPolygon.addLineTo({25, 75});
    expectedPolygon.addLineTo({25, 25});
    expectedPolygon.addEndPoly();

    auto checkFeatureGeometry = [&datasetHolder, &params] (
            const std::string& sublayerName,
            const mr::base::Vertices& expectedGeometry) {
        params.auxParams[maps::wiki::renderer::SUBLAYERS] = sublayerName;
        mr::data_set::ViewQueryContext ctx(&params);
        std::vector<mr::data_set::LayerView> views = datasetHolder->queryView(ctx);
        EXPECT_EQ(views.size(), 1u);
        {
            auto iter = views[0].iterator();
            EXPECT_TRUE(iter->hasNext());
            auto& f = iter->next();
            EXPECT_EQ(f.geom().shapes(), expectedGeometry);
        }
    };

    EXPECT_NO_THROW(checkFeatureGeometry("errors1", expectedPoint));

    EXPECT_NO_THROW(checkFeatureGeometry("errors2", expectedPoint));

    EXPECT_NO_THROW(checkFeatureGeometry("errors_polygon", expectedPolygon));
}

}; // Y_UNIT_TEST_SUITE geometry_expression_tests

} // namespace maps::wiki::tests
