#include "renderer.h"
#include "searchpath.h"
#include "filters.h"
#include "dbwrapper.h"
#include "sublayerstraverse.h"
#include <maps/libs/common/include/profiletimer.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/pgpool/include/pgpool3.h>
#include <maps/libs/xml/include/xml.h>
#include <maps/infra/yacare/include/error.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/threadutils/executor.h>
#include <yandex/maps/wiki/threadutils/threadpool.h>

#include <boost/format.hpp>

#include <fstream>
#include <iomanip>
#include <condition_variable>
#include <mutex>
#include <stack>
#include <vector>

namespace p3 = maps::pgpool3;


namespace maps::wiki::renderer {

class TileRendererHolder;

class FreeTileRenderers
{
public:
    typedef std::unique_ptr<TileRenderer> TileRendererPtr;

    void push(TileRendererPtr renderer)
    {
        std::lock_guard<std::mutex> lock(mutex_);
        data_.emplace(std::move(renderer));
        condition_.notify_one();
    }

private:
    friend class TileRendererHolder;

    TileRendererPtr pop()
    {
        std::unique_lock<std::mutex> lock(mutex_);
        while (data_.empty()) {
            condition_.wait(lock);
        }
        auto renderer = std::move(data_.top());
        data_.pop();
        return renderer;
    }

private:
    std::mutex mutex_;
    std::condition_variable condition_;
    std::stack<TileRendererPtr> data_;
};

class TileRendererHolder : boost::noncopyable
{
public:
    explicit TileRendererHolder(FreeTileRenderers& freeList)
        : freeList_(freeList)
    {}

    ~TileRendererHolder()
    {
        if (renderer_) {
            freeList_.push(std::move(renderer_));
        }
    }

    TileRenderer* get()
    {
        if (!renderer_) {
            renderer_ = freeList_.pop();
        }
        return renderer_.get();
    }

private:
    FreeTileRenderers& freeList_;
    FreeTileRenderers::TileRendererPtr renderer_;
};


namespace {

struct SubLayersInfo
{
    ParsedSubLayers parsedSubLayers;
    Scope scope;
};

class SplitSubLayers
{
public:
    SplitSubLayers(
        const std::string& subLayers,
        const LayerIds& textLayerIds,
        const std::set<Scope>& scopes)
    {
        if (subLayers.empty() || scopes.empty()) {
            return;
        }

        auto parsedSL = parseSubLayers(subLayers);
        if (scopes.count(Scope::All)) {
            ASSERT(scopes.size() == 1);
            addOrderedInfo(std::move(parsedSL), Scope::All);
            return;
        }

        ParsedSubLayers textSL;
        for (auto it = parsedSL.begin(); it != parsedSL.end(); ) {
            if (textLayerIds.count(it->first)) {
                textSL.emplace(it->first, std::move(it->second));
                it = parsedSL.erase(it);
            } else {
                ++it;
            }
        }

        if (scopes.count(Scope::Objects)) {
            addOrderedInfo(std::move(parsedSL), Scope::Objects);
        }
        if (scopes.count(Scope::Labels)) {
            addOrderedInfo(std::move(textSL), Scope::Labels);
        }
    }

    const std::vector<SubLayersInfo>& orderedData() const { return orderedData_; }

private:
    void addOrderedInfo(ParsedSubLayers&& parsedSubLayers, Scope scope)
    {
        if (!parsedSubLayers.empty()) {
            orderedData_.emplace_back(SubLayersInfo{std::move(parsedSubLayers), scope});
        }
    }

    std::vector<SubLayersInfo> orderedData_;
};

void mergeTiles(
    tilerenderer4::RenderedTile& tileTop,
    const tilerenderer4::RenderedTile& tile)
{
    if ((tileTop.outputDataSize != tile.outputDataSize) ||
        (tileTop.outputDataSize % 4))
    {
        ERROR() << "TILE SIZE : " << tileTop.outputDataSize << " : " << tile.outputDataSize;
        return;
    }

    auto* ptrTop = tileTop.outputData.get();
    const auto* ptr = tile.outputData.get();
    for (auto size = tile.outputDataSize; size; size -= 4, ptrTop += 4, ptr += 4) {
        if (!ptrTop[3] && ptr[3]) {
            //ptrTop[0] = ptr[0];
            //ptrTop[1] = ptr[1];
            //ptrTop[2] = ptr[2];
            //ptrTop[3] = ptr[3];
            *(reinterpret_cast<int32_t*>(ptrTop)) = *(reinterpret_cast<const int32_t*>(ptr));
        }
    }
}


class TransactionHolder: public renderer5::postgres::IPQXXTransactionHolder
{
public:
    explicit TransactionHolder(p3::ConnectionHandle&& connectionHandle)
        : transaction_(p3::makeNonTransaction(std::move(connectionHandle)))
    {}

    renderer5::postgres::PQXXTransaction& get() override { return *transaction_; }

private:
    p3::TransactionHandle transaction_;
};

std::string stripSpaces(const std::string& str)
{
    std::string result;
    for (auto ch : str) {
        if (ch != ' ') {
            result += ch;
        }
    }
    return result;
}

class TransactionProvider:
    public renderer5::postgres::PostgresTransactionProviderEx
{
public:
    TransactionProvider(
        DbWrapper& dbWrapper,
        const SearchPath& searchPath,
        const RequestParams& reqParams,
        Scope scope)
    {
        static const std::string SEARCH_PATH = "search_path";
        auto connection = dbWrapper.getConnection(reqParams.token, reqParams.branchId, scope);
        hostname_ = connection.get().hostname();

        transactionHolder_ = renderer5::postgres::PQXXTransactionHolderPtr(
            new TransactionHolder(std::move(connection)));

        auto path = searchPath.resolve(reqParams.branchId);
        if (!path.empty()) {
            auto&& txn = transactionHolder_->get();
            auto oldSearchPath = txn.get_variable(SEARCH_PATH);
            if (stripSpaces(oldSearchPath) != stripSpaces(path)) {
                if (!oldSearchPath.empty()) {
                    INFO() << "Overwriting search path. Old: " << oldSearchPath << "; New: " << path;
                }
                txn.set_variable(SEARCH_PATH, path);
            }
        }
        setQueryNamePrefix(std::to_string(reqParams.branchId));
    }

    renderer5::postgres::PQXXTransactionHolderPtr getTransaction() override { return transactionHolder_; }

    const std::string& hostname() const { return hostname_; }

private:
    std::string hostname_;
    renderer5::postgres::PQXXTransactionHolderPtr transactionHolder_;
};

} // namespace

/*
* Logger implementation
*/
/*
*Implementation of ILogger
*/
class Logger: public maps::renderer::base::ILogger
{
public:
    MessageType severity() const override
    {
        return ErrorMessage;
    }

    void setSeverity(MessageType /*severity*/) override {}

    void log(const std::wstring & message, MessageType subject) override
    {
        std::string m;
        m.assign(message.begin(),message.end());
        switch(subject){
        case ErrorMessage: ERROR() << m; break;
        case WarningMessage: WARN() << m; break;
        case InfoMessage: INFO() << m; break;
        default:
            DEBUG() << m;
        }
    }
};

/*
* Renderer implementation
*/

namespace{
/*Helper to walk around multidesign specific*/

    const xml3::Node extractNode(
            const xml3::Doc& config,
            const std::string& nodeName,
            const std::string& layerName ){
        auto commonXPath = str( boost::format("/config/%1%") % nodeName);
        auto mdXPath = str( boost::format("/config/multidesign/layer[@name='%1%']/%2%")
                            % layerName
                            % nodeName );
        auto node = config.node(commonXPath, true);
        if (node.isNull()){
            node = config.node(mdXPath);
        }
        return node;
    }
}

Renderers Renderer::create(
    const std::string& layerName,
    const std::filesystem::path& layerPath){

    Renderers renderers;
    xml3::Doc xml( (layerPath / "config.xml").string() );
    auto multiDesignLayerNodes = xml.nodes("/config/multidesign/layer", true);
    if (!multiDesignLayerNodes.size()) {
        renderers.insert(
            std::make_pair(
                layerName,
                RendererPtr(new Renderer(layerName, layerPath))));
    } else {
        for (size_t i = 0; i < multiDesignLayerNodes.size(); ++i) {
            auto multiDesignLayerName =
                multiDesignLayerNodes[i].attr<std::string>("name");
            renderers.insert(
                std::make_pair(
                    multiDesignLayerName,
                    RendererPtr(new Renderer(multiDesignLayerName, layerPath))));
        }
    }
    return renderers;
}

Renderer::Renderer(
    const std::string& layerName,
    const std::filesystem::path& layerPath)
    : layerName_(layerName)
{
    xml3::Doc xml((layerPath / "config.xml").string());
    mapFilename_ = (layerPath / "map.xml").string();

    auto optNode = extractNode(xml, "options", layerName);
    REQUIRE(
        !optNode.attr<bool>("layerDisabled", false),
        "Layer: " << layerName << " disabled");
    skipOutputCreation_ = optNode.attr<bool>("skipOutputCreation", false);
    expires_ = std::chrono::seconds(optNode.attr<unsigned int>("expires", 0));

    auto imageNode = extractNode(xml, "image", layerName);
    transparent_ = imageNode.attr<bool>("transparent");

    auto filtersNode = xml.node("/config/filters", true);
    if (!filtersNode.isNull()) {
        filters_.reset(new Filters(filtersNode));
    }

    searchPath_.reset(new SearchPath(xml.node("/config/searchPath", true)));

    auto limitNode = xml.node("/config/objectsQuantityLimit", true);
    objectsQuantityLimit_ =
        limitNode.isNull()
            ? DEFAULT_OBJECTS_QUANTITY_LIMIT
            : limitNode.attr<size_t>("value", DEFAULT_OBJECTS_QUANTITY_LIMIT);

    dbWrapper_.reset(new DbWrapper(xml.nodes("/config/db"), layerName_));

    freeTileRenderers_.reset(new FreeTileRenderers());

    // hack to avoid crash on create tile renderers in threads
    freeTileRenderers_->push(createTileRenderer());

    threadPool_.reset(new ThreadPool(THREADS));
    threadPool_->push([this] {
        for (auto i = THREADS - 1; i--;) {
            freeTileRenderers_->push(createTileRenderer());
        }
    });
    threadPool_->push([this] {
        for (auto i = THREADS; i--;) {
            freeTileRenderers_->push(createTileRenderer());
        }
    });

    TextLayersTraverse textTraverse;
    TileRendererHolder holder(*freeTileRenderers_);
    holder.get()->traverse(textTraverse);
    textLayerIds_ = textTraverse.textGroupLayerIds();
    INFO() << "Renderer " << layerName_
           << " text layers: " << common::join(textLayerIds_, ',');
}

Renderer::~Renderer()
{
    if (threadPool_) {
        threadPool_->shutdown();
    }
    dbWrapper_.reset(nullptr);
}

std::unique_ptr<TileRenderer>
Renderer::createTileRenderer() const
{
    ProfileTimer pt;
    maps::renderer::base::ILoggerPtr pLogger(new Logger);
    std::unique_ptr<TileRenderer>
        pTileRenderer( tilerenderer4::createOnlineTileRenderer(pLogger) );
    REQUIRE(pTileRenderer.get(), "ITileRenderer creation failed");

    pTileRenderer->open(mapFilename_);
    pTileRenderer->setOutputCreation(skipOutputCreation_);
    pTileRenderer->setSkipEmptyTiles(transparent_);
    INFO() << "Renderer " << layerName_ << ": IOnlineRenderer created... " << pt.getElapsedTime();
    return pTileRenderer;
}

struct Renderer::RenderParams
{
    const RequestParams& reqParams;
    const SubLayersInfo& subLayersInfo;
    tilerenderer4::OutputFormat outputFormat;
};

struct Renderer::RenderResult
{
    tilerenderer4::ReusableRenderedTile tile;
    std::string profileMsg;
    Scope scope;

    std::string profileMessage() const
    {
        return " " + toString(scope) + ": " + profileMsg;
    }
};

Renderer::RenderResult
Renderer::renderInternal(const RenderParams& params)
{
    ProfileTimer profileTimer;
    TileRendererHolder holder(*freeTileRenderers_);
    auto pTileRenderer = holder.get();

    const auto& reqParams = params.reqParams;
    const auto scope = params.subLayersInfo.scope;

    SubLayersTraverse traverse(
        params.subLayersInfo.parsedSubLayers, reqParams.token, *pTileRenderer,
        scope == Scope::Labels ? nullptr : filters_.get());

    std::unique_ptr<TransactionProvider> transactionProvider(
        new TransactionProvider(*dbWrapper_, *searchPath_, reqParams, scope));

    transactionProvider->setObjectsQuantityLimit(objectsQuantityLimit_);

    auto acquireTransactionDurationTime = profileTimer.getElapsedTime();
    auto hostname = transactionProvider->hostname();

    ProfileTimer renderProfileTimer;

    renderer5::postgres::PostgresTransactionProviderExPtr
        tProvider(transactionProvider.release());

    const double matrix[] = {1.0, 0.0, 0.0, 1.0, reqParams.dx, reqParams.dy};

    tilerenderer4::ReusableRenderedTile tile =
        pTileRenderer->render(
            reqParams.x, reqParams.y, reqParams.z,
            params.outputFormat,
            tProvider,
            1.0, // scale
            matrix,
            traverse);

    std::ostringstream os;
    os  << tile.outputDataSize
        << " acquire=" << acquireTransactionDurationTime
        << " render=" << renderProfileTimer.getElapsedTime()
        << " hostname=" << hostname;

    return RenderResult{std::move(tile), os.str(), scope};
}

RenderedData
Renderer::render(const RequestParams& reqParams)
{
    DEBUG() << "Renderer::render " << reqParams.reqStr;

    if (!dbWrapper_->isBranchAllowed(reqParams.branchId)) {
        throw yacare::errors::Forbidden() <<
            "Forbidden access to branch: " << reqParams.branchId;
    }

    ProfileTimer profileTimer;
    std::string profileMessage;
    auto wrapResult = [&](RenderedData&& rdata) {
        if (!profileMessage.empty()) {
            INFO() << "RENDER PROFILE " << profileTimer.getElapsedTime()
                   << profileMessage
                   << " : output: " << rdata.size
                   << " : " << reqParams.reqStr;
        }
        return std::move(rdata);
    };

    auto allowedScopes = dbWrapper_->allowedScopes(reqParams.branchId);
    SplitSubLayers splitResult(reqParams.subLayers, textLayerIds_, allowedScopes);
    const auto& orderedData = splitResult.orderedData();
    if (orderedData.empty()) {
        return wrapResult(RenderedData());
    }

    auto outputFormatPng =
        transparent_ ? tilerenderer4::OutputFormatRgbaPng
                     : tilerenderer4::OutputFormatPalettePng;

    if (orderedData.size() == 1) {
        auto renderResult = renderInternal({reqParams, orderedData.front(), outputFormatPng});
        auto& tile = renderResult.tile;
        profileMessage = renderResult.profileMessage();
        return wrapResult(RenderedData(std::move(tile.outputData), tile.outputDataSize));
    }

    auto renderAll = [&] {
        typedef std::unique_ptr<Renderer::RenderResult> RenderResultPtr;

        std::vector<RenderResultPtr> renderResults;
        renderResults.resize(orderedData.size());

        Executor executor;
        for (size_t index = 0; index < orderedData.size(); ++index) {
            executor.addTask([&, index] {
                renderResults[index].reset(new RenderResult(renderInternal(
                    {reqParams, orderedData[index], tilerenderer4::OutputFormatRgbaRaw})));
            });
        }
        executor.executeAll(*threadPool_);

        std::vector<RenderResultPtr> nonEmptyResults;
        for (auto& renderResult : renderResults) {
            if (renderResult) {
                profileMessage += renderResult->profileMessage();
                if (renderResult->tile.outputDataSize) {
                    nonEmptyResults.emplace_back(std::move(renderResult));
                }
            }
        }
        return nonEmptyResults;
    };

    auto results = renderAll();
    if (results.empty()) {
        return wrapResult(RenderedData());
    }

    ProfileTimer pt;
    auto& top = results.back();
    for (size_t index = results.size() - 1; index--; ) {
        const auto& renderResult = results[index];
        mergeTiles(top->tile, renderResult->tile);
    }

    auto composeTask = tilerenderer4::ComposeTask::create();
    composeTask->outputFormat = outputFormatPng;

    auto res = tilerenderer4::composeTiles({std::move(top->tile)}, *composeTask);
    profileMessage += " png: " + pt.getElapsedTime();

    return wrapResult(RenderedData(std::move(res.outputData), res.outputDataSize));
}

} //namespace maps::wiki::renderer
