#include "render.h"

#include <maps/libs/geolib/include/spatial_relation.h>
#include <yandex/maps/wiki/threadutils/threadpool.h>
#include <yandex/maps/wiki/threadutils/executor.h>
#include <maps/libs/tile/include/const.h>

using namespace maps::geolib3;
using namespace maps::tile;
using namespace maps::wiki;

namespace maps::wiki::renderer_overlay {

namespace {
const size_t TILE_THREADS_COUNT = 8;
const PIXELCOLOR ERROR_COLOR = 0xFFFF0000;//RED
};

Image renderTile(
    const maps::tile::Tile& mapTile,
    const OverlayImageInfoByZOrder& overlayImageInfoByZOrder,
    const std::optional<Polygon2>& cutOffPolygon)
{
    const auto tileDispBbox = mapTile.dispBox();
    auto tileDisplayCorner = tileDispBbox.lt();
    ThreadPool threadPool(TILE_THREADS_COUNT);
    Executor threadedExecutor;
    Image targetImage(maps::tile::TILE_SIZE, maps::tile::TILE_SIZE);
    for (size_t y = 0; y < maps::tile::TILE_SIZE; ++y) {
        auto renderRow = [&](size_t y) {
            for (size_t x = 0; x < maps::tile::TILE_SIZE; ++x) {
                Point2 tilePointDisplay(x + tileDisplayCorner.x(), y + tileDisplayCorner.y());
                if (cutOffPolygon && !spatialRelation(
                        *cutOffPolygon,
                        displayToMercator(
                            {
                                static_cast<unsigned long>(tilePointDisplay.x()),
                                static_cast<unsigned long>(tilePointDisplay.y())
                            },
                            mapTile.z()),
                        SpatialRelation::Contains))
                {
                    continue;
                }
                for (const auto& [_, overlayImageInfo] : overlayImageInfoByZOrder) {
                    if (!overlayImageInfo.transformation) {
                        targetImage.setPixelColor(x, y, ERROR_COLOR);
                        break;
                    }
                    const auto sourcePixel =
                        overlayImageInfo.transformation->toImageSpace(tilePointDisplay);
                    const auto& sourceImg = *overlayImageInfo.image;
                    if (sourcePixel.x() < 0 && sourcePixel.y() < 0) {
                        continue;
                    }
                    size_t srcX = sourcePixel.x();
                    size_t srcY = sourcePixel.y();
                    if (srcX < sourceImg.width() && srcY < sourceImg.height()) {
                        targetImage.setPixelColor(x, y, sourceImg.pixelColor(srcX, srcY));
                        break;
                    }
                }
            }
        };
        threadedExecutor.addTask([&, y]{ renderRow(y);});
    }
    //auto threadPool = cfg()->threadPools().pool(TILE_THREADS_COUNT);TODO Investigate segfault reason while using this pool
    threadedExecutor.executeAllInThreads(threadPool);
    return targetImage;
}
} // namespace maps::wiki::renderer_overlay
