#include "storage.h"
#include "config.h"
#include "image.h"

#include <maps/libs/s3mds/include/auth.h>
#include <maps/libs/s3mds/include/storage.h>
#include <maps/libs/log8/include/log8.h>
#include <util/generic/guid.h>


using namespace maps::s3mds;

namespace maps::wiki::renderer_overlay {

namespace {

using namespace std::literals::chrono_literals;

const size_t CACHE_SIZE = 100;
const std::chrono::milliseconds CACHE_TIME = 10min;

std::string
generateUUID()
{
    const auto tsGuid = CreateGuidAsString();
    return std::string(tsGuid.data(), tsGuid.length());
}

std::unique_ptr<Storage> s_storage_;

}  // namespace

Storage::Storage(const Config& cfg)
    : cfg_(cfg)
    , dataIdToDataCache_(CACHE_SIZE, CACHE_TIME)
{
}

std::string Storage::store(const Image& image)
{
    s3mds::Config s3mdsCfg(cfg_.s3EndPoint(), cfg_.s3Bucket());
    s3mds::AuthHolder auth =
        std::make_unique<s3mds::AccessKeyAuth>(cfg_.s3AccessKeyId(), cfg_.s3AccessKeySecret());
    s3mds::Storage storage(
        s3mdsCfg,
        std::move(auth));
    const auto uuid = generateUUID();
    storage.put(cfg_.s3SourceImagesFolder() + uuid, image.pngData());
    DEBUG() << " Stored new image: " << uuid;
    return uuid;
}

std::shared_ptr<Image> Storage::read(const std::string& dataId)
{
    auto cached = dataIdToDataCache_.get(dataId);
    if (cached) {
        return *cached;
    }
    AuthHolder auth( new AccessKeyAuth(cfg_.s3AccessKeyId(), cfg_.s3AccessKeySecret()));
    s3mds::Config s3mdsCfg(cfg_.s3EndPoint(), cfg_.s3Bucket());
    s3mds::Storage storage(
        s3mdsCfg,
        AuthHolder(new AccessKeyAuth(cfg_.s3AccessKeyId(), cfg_.s3AccessKeySecret())));
    auto data = storage.get(cfg_.s3SourceImagesFolder() + dataId);
    auto imgPtr = std::make_shared<Image>(data);
    dataIdToDataCache_.put(dataId, imgPtr);
    return imgPtr;
}


StorageScope::StorageScope(const Config& cfg)
{
    s_storage_.reset(new Storage(cfg));
}
StorageScope::~StorageScope()
{
    s_storage_.reset();
}

Storage* storage()
{
    ASSERT(s_storage_);
    return s_storage_.get();
}

} // namespace maps::wiki::renderer_overlay
