#include <maps/wikimap/mapspro/services/social/src/api/globals.h>
#include <maps/wikimap/mapspro/services/social/src/libs/common/common.h>
#include <maps/wikimap/mapspro/services/social/src/libs/yacare/helpers.h>
#include <maps/wikimap/mapspro/services/social/src/libs/assessment/serialize.h>

#include <maps/wikimap/mapspro/libs/assessment/include/gateway.h>

namespace maps::wiki::socialsrv {

using namespace social;
using namespace assessment;

namespace {

void checkPermission(UserId uid, Entity::Domain domain, Qualification qualification)
{
    const std::string domainStr(toString(domain));
    const std::string qualificationStr(toString(qualification));
    const auto permission = "mpro/social/assessment/" + domainStr + "/grade/" + qualificationStr;

    Globals::aclChecker().checkPermission(uid, permission);
}

void checkPermission(assessment::Gateway& gateway, TUid uid, TId sampleId)
{
    const auto sampleInfo = gateway.findSample(sampleId);
    checkPermission(uid, sampleInfo.entityDomain, sampleInfo.qualification);
}

SampleFeed getSampleFeed(
    TUid uid,
    const std::string& token,
    const SampleFeedParams& params,
    const SampleFilter& filter)
{
    auto socialTxn = Globals::dbPools().socialReadTxn(token);
    checkPermission(uid, *filter.entityDomain(), *filter.qualification());

    auto console = assessment::Gateway(*socialTxn).console(uid);
    return console.sampleFeed(params, filter);
}

SampleInfo getSampleInfo(
    TUid uid,
    const std::string& token,
    TId sampleId)
{
    auto socialTxn = Globals::dbPools().socialReadTxn(token);
    assessment::Gateway gateway(*socialTxn);

    auto sampleInfo = gateway.findSample(sampleId);
    checkPermission(uid, sampleInfo.entityDomain, sampleInfo.qualification);
    return sampleInfo;
}

std::optional<Unit> acquireUnit(
    TUid uid,
    TId sampleId,
    SkipAcquired skipAcquired)
{
    auto writeContext = Globals::dbPools().writeContext();

    assessment::Gateway gateway(writeContext.socialTxn());
    checkPermission(gateway, uid, sampleId);

    auto unit = gateway.console(uid).acquireUnit(sampleId, skipAcquired);
    writeContext.commit();

    return unit;
}

void releaseUnit(
    TUid uid,
    TId sampleId)
{
    auto writeContext = Globals::dbPools().writeContext();

    assessment::Gateway gateway(writeContext.socialTxn());
    checkPermission(gateway, uid, sampleId);

    gateway.console(uid).releaseUnit(sampleId);
}

constexpr long THREADS_COUNT = 1;
constexpr long BACKLOG = 512;
yacare::ThreadPool assessmentSamplesThreadPool("assessmentSamplesThreadPool", THREADS_COUNT, BACKLOG);

} // namespace

YCR_USE(assessmentSamplesThreadPool) {

YCR_RESPOND_TO("GET /assessment/samples", uid, token = "")
{
    const auto entityDomain = enum_io::fromString<Entity::Domain>(
        queryParam<std::string>(request, "entity-domain"));
    const auto qualification = enum_io::fromString<Qualification>(
        queryParam<std::string>(request, "qualification"));

    const auto graded = optionalQueryParam<bool>(request, "graded");
    const auto createdBefore = parseOptionalTimePoint(request, "created-before");
    const auto createdAfter = parseOptionalTimePoint(request, "created-after");
    const auto before = queryParam(request, "before", 0);
    const auto after = queryParam(request, "after", 0);
    const auto perPage = queryParam(request, "per-page", SampleFeedParams::perPageDefault);

    auto filter = SampleFilter()
        .entityDomain(entityDomain)
        .qualification(qualification);

    if (graded) {
        filter.graded(*graded);
    }
    if (createdBefore || createdAfter) {
        filter.createdAt(DateTimeCondition(createdAfter, createdBefore));
    }

    auto sampleFeed = getSampleFeed(uid, token, SampleFeedParams(before, after, perPage), filter);
    makeJsonResponse(response, toJson(sampleFeed));
}

YCR_RESPOND_TO("GET /assessment/samples/$", uid, token = "")
{
    const auto sampleId = positionalParam<TId>(argv, 0);
    makeJsonResponse(response, toJson(getSampleInfo(uid, token, sampleId)));
}

YCR_RESPOND_TO("POST /assessment/samples/$/acquire-unit", uid)
{
    const auto sampleId = positionalParam<TId>(argv, 0);
    const auto skip = queryParam<bool>(request, "force", false);

    const auto unit = acquireUnit(uid, sampleId, skip ? SkipAcquired::Yes : SkipAcquired::No);
    makeJsonResponse(response, (unit? toJson(*unit) : "null"));
}

YCR_RESPOND_TO("POST /assessment/samples/$/release-unit", uid)
{
    const auto sampleId = positionalParam<TId>(argv, 0);
    releaseUnit(uid, sampleId);
}

} // YCR_USE

} // namespace maps::wiki::socialsrv
