#include <maps/wikimap/mapspro/services/social/src/libs/assessment/schema.h>

#include <maps/wikimap/mapspro/services/social/src/api/globals.h>
#include <maps/wikimap/mapspro/services/social/src/libs/common/common.h>
#include <maps/wikimap/mapspro/services/social/src/libs/yacare/helpers.h>
#include <maps/wikimap/mapspro/services/social/src/libs/assessment/serialize.h>

#include <maps/wikimap/mapspro/libs/social/include/yandex/maps/wiki/social/feedback/gateway_ro.h>
#include <maps/wikimap/mapspro/libs/assessment/include/gateway.h>

#include <fmt/format.h>

namespace maps::wiki::socialsrv {

using namespace social;
using namespace assessment;

namespace {

constexpr long THREADS_COUNT = 1;
constexpr long BACKLOG = 512;
yacare::ThreadPool assessmentThreadPool("assessmentThreadPool", THREADS_COUNT, BACKLOG);

std::optional<Qualification> getAllowedGradeQualification(UserId uid, Entity::Domain domain)
{
    const std::string domainStr(toString(domain));
    const std::string expertStr(toString(Qualification::Expert));
    const auto permission = "mpro/social/assessment/" + domainStr + "/grade";
    auto& checker = Globals::aclChecker();

    if (!checker.userHasPartOfPermission(uid, permission)) {
        return {};
    } else if (checker.userHasPermission(uid, permission + "/" + expertStr)) {
        return Qualification::Expert;
    } else {
        return Qualification::Basic;
    }
}

bool isAllowedToViewGradesReceivedBy(UserId uid, Entity::Domain domain, const std::string& who)
{
    const std::string domainStr(toString(domain));
    const auto permission = "mpro/social/assessment/" + domainStr + "/view/received-by-" + who;
    return Globals::aclChecker().userHasPermission(uid, permission);
}

bool isAllowedToViewAssessorsIdentity(UserId uid, Entity::Domain domain)
{
    const std::string domainStr(toString(domain));
    const auto permission = "mpro/social/assessment/" + domainStr + "/view/assessors-identity";
    return Globals::aclChecker().userHasPermission(uid, permission);
}

UnitFeed hideGradedByNotEqualTo(const UnitFeed& feed, UserId uid)
{
    GradedUnitVec gradedUnits = feed.units();
    for (auto& gradedUnit : gradedUnits) {
        for (auto& grade : gradedUnit.grades) {
            if (grade.gradedBy != uid) {
                grade.gradedBy = 0;
            }
        }
    }

    return UnitFeed(std::move(gradedUnits), feed.hasMore());
}

GradesVisibility gradesVisibility(UserId uid, Entity::Domain domain)
{
    const auto receivedByMe = isAllowedToViewGradesReceivedBy(uid, domain, "me");
    const auto receivedByOthers = isAllowedToViewGradesReceivedBy(uid, domain, "others");

    if (receivedByMe && receivedByOthers) {
        return GradesVisibility::All;
    }
    if (receivedByMe) {
        return GradesVisibility::MyOrReceivedByMe;
    }

    if (receivedByOthers) {
        throw yacare::errors::UnprocessableEntity()
            << "User " << uid << " has received-by-others permission for " << domain
            << ", yet is missing according received-by-me permission";
    }

    SOCIAL_REQUIRE(
        !getAllowedGradeQualification(uid, domain),
        Forbidden,
        "User " << uid << " is prohibited to view any " << domain << " grades");

    return GradesVisibility::My;
}

void checkRateUnitsGrade(auto& socialTxn, UserId uid, auto entityDomain)
{
    if (Globals::aclChecker().userHasPermission(uid, "mpro/tools/unlimited-assessment-rate")) {
        return;
    }

    auto check = [&] (const std::optional<std::chrono::seconds>& interval, auto context) {
        if (!interval) {
            return;
        }

        static const std::string ERROR_MESSAGE = "Too many assessment operations";

        WARN() << ERROR_MESSAGE << ", uid: " << uid << " : " << context << "." << interval->count();

        throw Error(Error::Status::AssessmentLimitExceeded) << ERROR_MESSAGE;
    };

    const auto& rateLimiter = Globals::assessmentRateLimiter();
    check(rateLimiter.checkLimitExceeded(socialTxn, uid, entityDomain), entityDomain);
    check(rateLimiter.checkTotalLimitExceeded(socialTxn, uid), "total");
}

} // namespace

YCR_USE(assessmentThreadPool) {

YCR_RESPOND_TO("POST /assessment/units/grade", uid)
{
    const auto parsedRequest = parsePostGradeRequest(request.body());
    SOCIAL_REQUIRE(
        uid != parsedRequest.action.by,
        InvalidOperation,
        "Self-assessment is prohibited");

    const auto uidQualification = getAllowedGradeQualification(uid, parsedRequest.entity.domain);
    SOCIAL_REQUIRE(
        uidQualification,
        Forbidden,
        "User " << uid << " is prohibited to grade " << parsedRequest.entity.domain);

    auto writeContext = Globals::dbPools().writeContext();
    auto& socialTxn = writeContext.socialTxn();
    checkRateUnitsGrade(socialTxn, uid, parsedRequest.entity.domain);

    assessment::Gateway gateway(socialTxn);
    auto console = gateway.console(uid);

    gateway.checkEntityAction(parsedRequest.entity, parsedRequest.action);
    auto unitId = gateway.getOrCreateUnit(parsedRequest.entity, parsedRequest.action);

    auto grade = console.gradeUnit(
        unitId,
        parsedRequest.gradeValue,
        parsedRequest.comment,
        *uidQualification);

    auto token = writeContext.commit();
    makeJsonResponse(response, toJson(PostGradeResult{grade, token}));
}

YCR_RESPOND_TO("POST /assessment/units/$/fix", uid)
{
    const auto unitId = positionalParam<TId>(argv, 0);
    auto writeContext = Globals::dbPools().writeContext();
    auto& socialTxn = writeContext.socialTxn();

    SOCIAL_REQUIRE(
        assessment::Gateway(socialTxn).console(uid).tryFixUnit(unitId),
        Forbidden,
        "Fixing unit " << unitId << " is forbidden for user " << uid);

    const auto token = writeContext.commit();
    makeJsonResponse(response, toJson(TokenResult{token}));
}

YCR_RESPOND_TO("POST /assessment/units/$/accept-grade-refutation", uid)
{
    const auto unitId = positionalParam<TId>(argv, 0);
    auto writeContext = Globals::dbPools().writeContext();
    auto& socialTxn = writeContext.socialTxn();

    SOCIAL_REQUIRE(
        assessment::Gateway(socialTxn).console(uid).tryAcceptRefutation(unitId),
        Forbidden,
        "Accepting refutation for unit " << unitId << " is forbidden for user " << uid);

    const auto token = writeContext.commit();
    makeJsonResponse(response, toJson(TokenResult{token}));
}

YCR_RESPOND_TO("GET /assessment/units", uid, token = "")
{
    const auto entityDomain = enum_io::fromString<Entity::Domain>(
        queryParam<std::string>(request, "entity-domain"));

    const auto entityIds = optionalVectorQueryParam<std::string>(request, "entity-id");
    const auto actionBy = optionalQueryParam<TUid>(request, "action-by");
    const auto graded = optionalQueryParam<bool>(request, "graded");
    const auto gradedBy = optionalQueryParam<TUid>(request, "graded-by");
    const auto gradeValue = optionalQueryParam<std::string>(request, "grade-value");
    const auto gradeConfirmed = optionalQueryParam<bool>(request, "grade-confirmed");
    const auto gradeStatus = optionalQueryParam<std::string>(request, "grade-status");
    const auto status = optionalQueryParam<std::string>(request, "status");

    const auto beforeId = queryParam<ID>(request, "before", 0);
    const auto afterId = queryParam<ID>(request, "after", 0);
    const auto perPage = queryParam<size_t>(request, "per-page", UnitFeedParams::perPageDefault);

    auto filter = UnitFilter().entityDomain(entityDomain);
    if (entityIds) {
        filter.entityIds({entityIds->cbegin(), entityIds->cend()});
    }
    if (actionBy) {
        filter.actionBy(*actionBy);
    }
    if (graded) {
        filter.graded(*graded);
    }
    if (gradedBy) {
        filter.gradedBy(*gradedBy);
    }
    if (gradeValue) {
        filter.gradeValue(enum_io::fromString<Grade::Value>(*gradeValue));
    }
    if (gradeConfirmed) {
        filter.gradeConfirmed(*gradeConfirmed);
    }
    if (gradeStatus) {
        filter.gradeStatus(enum_io::fromString<GradeStatus>(*gradeStatus));
    }
    if (status) {
        filter.status(enum_io::fromString<UnitStatus>(*status));
    }

    UnitFeedParams params(beforeId, afterId, perPage);

    auto socialTxn = Globals::dbPools().socialReadTxn(token);
    auto console = assessment::Gateway(*socialTxn).console(uid);
    auto feed = console.unitFeed(params, filter, gradesVisibility(uid, entityDomain));

    if (!isAllowedToViewAssessorsIdentity(uid, entityDomain)) {
        feed = hideGradedByNotEqualTo(feed, uid);
    }
    makeJsonResponse(response, toJson(feed));
}

} // YCR_USE

} // namespace maps::wiki::socialsrv
