#include <maps/wikimap/mapspro/services/social/src/libs/stats/rating_common.h>
#include <maps/wikimap/mapspro/services/social/src/libs/stats/scroll_window.h>
#include <maps/wikimap/mapspro/services/social/src/api/globals.h>
#include <maps/wikimap/mapspro/services/social/src/libs/yacare/helpers.h>

#include <maps/libs/json/include/builder.h>

#include <pqxx/pqxx>
#include <algorithm>
#include <string>
#include <tuple>

namespace srv = maps::wiki::socialsrv;

namespace maps::wiki::socialsrv {

namespace {

struct RatingMetaInfo
{
    size_t version;
    size_t size;
};

RatingMetaInfo fetchMetaInfo(
        pqxx::transaction_base& txn,
        RatingType ratingType)
{
    // N.B.: using V2 version of rating
    //
    auto metaInfo = txn.exec(
        "SELECT version, size FROM rating.meta_v2"
            " WHERE type = " + txn.quote(std::string(toString(ratingType))));
    ASSERT(metaInfo.size() == 1);

    return {
        metaInfo[0]["version"].as<size_t>(),
        metaInfo[0]["size"].as<size_t>()};
}

yacare::ThreadPool ratingThreadPool("ratingThreadPool", 1, 1024);

} // namespace

YCR_RESPOND_TO("GET /rating/$", YCR_IN_POOL(ratingThreadPool))
{
    auto ratingTypeStr = positionalParam<std::string>(argv, 0);
    auto anchorUid = queryParam<UserId>(request, "anchor-uid", 0);
    auto before =
        optionalQueryParam<size_t>(request, "before");
    auto after =
        optionalQueryParam<size_t>(request, "after");
    auto windowSize = queryParam<size_t>(request, "per-page", 10);

    RatingType ratingType;
    REQUIRE(tryFromString(ratingTypeStr, ratingType), yacare::errors::BadRequest() << "Invalid rating type");

    REQUIRE(windowSize, yacare::errors::BadRequest() << "Invalid window size");
    REQUIRE(anchorUid || before || after, yacare::errors::BadRequest() << "Missing anchor parameter");

    auto socialTxn = Globals::dbPools().socialRepeatableReadTxn();
    auto metaInfo = fetchMetaInfo(*socialTxn, ratingType);

    // NOTE: rating positions are 1-based, need to be shifted
    ScrollWindow window;
    bool addDummy = false;
    if (anchorUid) {
        size_t anchorPos =
            getRatingPos(*socialTxn, ratingType, anchorUid);

        if (anchorPos <= metaInfo.size) {
            window = ScrollWindow(
                windowSize,
                metaInfo.size,
                anchorPos - 1, ScrollWindow::Align::Center);
        } else { // user has no rating, adding dummy row at the end
            addDummy = true;
            window = ScrollWindow(
                windowSize - 1,
                metaInfo.size,
                metaInfo.size - 1, ScrollWindow::Align::Right);
        }
    } else if (before) {
        *before = std::min(*before, metaInfo.size);

        if (*before > 1) {
            window = ScrollWindow(
                windowSize,
                metaInfo.size,
                *before - 2,
                ScrollWindow::Align::Right);
        } else {
            window = ScrollWindow(
                0, metaInfo.size, 0, ScrollWindow::Align::Right);
        }
    } else if (after) {
        window = ScrollWindow(
            windowSize,
            metaInfo.size,
            *after,
            ScrollWindow::Align::Left);
    }

    // N.B.: using V2 version of rating
    //
    auto ratingRows = socialTxn->exec(
            "SELECT * FROM rating." + std::string(toString(ratingType)) + "_v2" +
                " ORDER BY pos"
                " OFFSET " + std::to_string(window.offset()) +
                " LIMIT " + std::to_string(window.count()));

    response << YCR_JSON(respBuilder) {
        respBuilder["rating"] << YCR_JSON_ARRAY(ratingsBuilder) {
            for (const auto& row : ratingRows) {
                ratingsBuilder << YCR_JSON(ratingBuilder) {
                    ratingBuilder["pos"] = row["pos"].as<size_t>();
                    ratingBuilder["uid"] = row["uid"].as<std::string>();
                    ratingBuilder["score"] = row["score"].as<double>();
                };
            }
            if (addDummy) {
                ASSERT(anchorUid);
                ratingsBuilder << YCR_JSON(ratingBuilder) {
                    ratingBuilder["pos"] = metaInfo.size + 1;
                    ratingBuilder["uid"] = std::to_string(anchorUid);
                    ratingBuilder["score"] = 0.0;
                };
            }
        };
        respBuilder["hasBefore"] = window.hasBefore();
        respBuilder["hasAfter"] = window.hasAfter();
        respBuilder["version"] = metaInfo.version;
    };
}

} // namespace maps::wiki::socialsrv
