#include "tasks_from_json.h"

#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <maps/wikimap/mapspro/services/social/src/libs/feedback/common.h>
#include <maps/wikimap/mapspro/services/social/src/libs/feedback/description.h>
#include <maps/wikimap/mapspro/services/social/src/libs/yacare/bbox.h>

#include <yandex/maps/wiki/social/feedback/attribute_names.h>
#include <yandex/maps/wiki/social/feedback/description_keys.h>
#include <yandex/maps/wiki/social/feedback/description_producers.h>
#include <yandex/maps/wiki/social/feedback/description_serialize.h>
#include <yandex/maps/wiki/social/feedback/enums.h>
#include <yandex/maps/wiki/social/feedback/types_back_compatibility.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/serialization.h>

#include <maps/libs/ymapsdf/include/ft.h>
#include <maps/libs/ymapsdf/include/rd.h>

#include <boost/lexical_cast.hpp>

using FtType = maps::ymapsdf::ft::Type;
using RoadDirection = maps::ymapsdf::rd::Direction;

namespace maps::wiki::socialsrv {

namespace mwsf = social::feedback;
namespace mwsfa = social::feedback::attrs;
namespace sf = social::feedback;

namespace {

std::set<std::string> userRoles(pqxx::transaction_base& txn, acl::UID uid)
{
    std::set<std::string> roles;
    try {
        acl::ACLGateway gtw(txn);
        auto user = gtw.user(uid);
        for (const auto& policy : user.allPolicies()) {
            roles.emplace(policy.role().name());
        }
    } catch (std::exception&) {
        // no code
    }
    return roles;
}

mwsf::TaskNew specificNewTaskFromJson(
        const json::Value& value,
        mwsf::Type type,
        mwsf::Description description,
        mwsf::Attrs specificAttrs = mwsf::Attrs())
{
    if (value.hasField(mwsfa::COMMIT_ID)) {
        specificAttrs.addCustom(
            mwsfa::COMMIT_ID, value[mwsfa::COMMIT_ID].as<std::string>());
    }

    if (value.hasField(mwsfa::OBJECT_DIFF)) {
        specificAttrs.add(
            mwsf::AttrType::ObjectDiff, value[mwsfa::OBJECT_DIFF]);
    }

    if (value.hasField(mwsfa::SOURCE_CONTEXT)) {
        specificAttrs.add(
            mwsf::AttrType::SourceContext, value[mwsfa::SOURCE_CONTEXT]);
    }

    if (value.hasField(jsids::USER_ATTRS)) {
        specificAttrs.add(mwsf::AttrType::UserData, value[jsids::USER_ATTRS]);
    }

    mwsf::TaskNew newTask(
        geolib3::convertGeodeticToMercator(
            geolib3::readGeojson<geolib3::Point2>(value[jsids::POSITION])
        ),
        type,
        value[jsids::SOURCE].as<std::string>(),
        std::move(description)
    );

    if (value.hasField(jsids::INDOOR_LEVEL)) {
        newTask.indoorLevel = value[jsids::INDOOR_LEVEL].as<std::string>();
    }

    newTask.attrs = std::move(specificAttrs);

    if (value.hasField(jsids::OBJECT_ID)) {
        newTask.objectId = boost::lexical_cast<social::TId>(
            value[jsids::OBJECT_ID].as<std::string>());
    }

    if (value.hasField(jsids::HIDDEN)) {
        newTask.hidden = value[jsids::HIDDEN].as<bool>();
    }

    if (value.hasField(jsids::INTERNAL_CONTENT)) {
        newTask.internalContent = value[jsids::INTERNAL_CONTENT].as<bool>();
    }

    return newTask;
}

} // namespace

mwsf::TaskNew poiNewTaskFromJson(const json::Value& value)
{
    auto actionString = value[mwsfa::SUGGESTED_ACTION].as<std::string>();
    auto action = boost::lexical_cast<mwsf::SuggestedAction>(actionString);
    mwsf::SuggestedActionDescr descr {action};
    mwsf::Attrs attrs;
    attrs.addCustom(mwsfa::SUGGESTED_ACTION, actionString);
    if (value.hasField(mwsfa::USER_COMMENT)) {
        attrs.addCustom(mwsfa::USER_DATA_COMMENT, value[mwsfa::USER_COMMENT].as<std::string>());
    }
    return specificNewTaskFromJson(
        value,
        mwsf::Type::Poi,
        descr.toDescription(),
        std::move(attrs)
    );
}

mwsf::TaskNew roadDirectionNewTaskFromJson(const json::Value& value)
{
    // extract params
    auto dirAsString = value[mwsfa::SUPPOSED_DIRECTION].as<std::string>();
    auto dir = boost::lexical_cast<RoadDirection>(dirAsString);

    // construct attributes
    mwsf::Attrs attrs;
    attrs.addCustom(mwsfa::SUPPOSED_DIRECTION, dirAsString);

    // construct description
    mwsf::WrongRoadDirectionDescr descr{dir};

    return specificNewTaskFromJson(
        value,
        mwsf::Type::RoadDirection,
        descr.toDescription(),
        std::move(attrs)
    );
}

mwsf::TaskNew speedLimitNewTaskFromJson(const json::Value& value)
{
    // extract params
    std::optional<std::string> dirAsString;
    std::optional<maps::ymapsdf::rd::Direction> dir;
    if (value.hasField(mwsfa::DIRECTION)) {
        dirAsString = value[mwsfa::DIRECTION].as<std::string>();
        dir = boost::lexical_cast<maps::ymapsdf::rd::Direction>(*dirAsString);
    }

    bool truck = false;
    if (value.hasField(mwsfa::TRUCK)) {
        truck = value[mwsfa::TRUCK].as<bool>();
    }
    bool confirmingSignNotFound = false;
    if (value.hasField(mwsfa::CONFIRMING_SIGN_NOT_FOUND)) {
        confirmingSignNotFound = value[mwsfa::CONFIRMING_SIGN_NOT_FOUND].as<bool>();
    }

    std::optional<int> currentSpeedLimit;
    if (value.hasField(mwsfa::CURRENT_SPEED_LIMIT)) {
        currentSpeedLimit = value[mwsfa::CURRENT_SPEED_LIMIT].as<int>();
    }
    int correctSpeedLimit = value[mwsfa::CORRECT_SPEED_LIMIT].as<int>();

    // construct attributes
    mwsf::Attrs attrs;
    if (currentSpeedLimit) {
        attrs.addCustom(
            mwsfa::CURRENT_SPEED_LIMIT, std::to_string(*currentSpeedLimit));
    }
    if (truck) {
        attrs.addCustom(
            mwsfa::TRUCK, "true");
    }
    if (confirmingSignNotFound) {
        attrs.addCustom(
            mwsfa::CONFIRMING_SIGN_NOT_FOUND, "true");
    }
    if (dirAsString) {
        attrs.addCustom(
            mwsfa::DIRECTION, *dirAsString);
    }
    attrs.addCustom(
        mwsfa::CORRECT_SPEED_LIMIT, std::to_string(correctSpeedLimit));

    // construct description
    mwsf::WrongSpeedLimitDescr descr{dir, truck, confirmingSignNotFound, currentSpeedLimit, correctSpeedLimit};

    return specificNewTaskFromJson(
        value,
        mwsf::Type::RoadSpeedLimit,
        descr.toDescription(),
        std::move(attrs)
    );
}

mwsf::TaskNew absentBarrierNewTaskFromJson(const json::Value& value)
{
    // construct attributes
    mwsf::Attrs attrs;

    std::optional<std::string> userComment;
    if (value.hasField(mwsfa::USER_COMMENT)) {
        userComment = value[mwsfa::USER_COMMENT].as<std::string>();
        attrs.addCustom(mwsfa::USER_COMMENT, *userComment);
        attrs.addCustom(mwsfa::USER_DATA_COMMENT, *userComment);
    }

    return specificNewTaskFromJson(
        value,
        mwsf::Type::Barrier,
        sf::DescriptionI18n(sf::tanker::fb_desc::ABSENT_BARRIER_KEY),
        std::move(attrs)
    );
}

mwsf::TaskNew absentEntranceNewTaskFromJson(const json::Value& value)
{
    // extract params
    auto entranceName = value[mwsfa::ENTRANCE_NAME].as<std::string>();

    // construct attributes
    mwsf::Attrs attrs;
    attrs.addCustom(mwsfa::ENTRANCE_NAME, entranceName);

    // construct description
    mwsf::AbsentEntranceDescr descr{entranceName};

    return specificNewTaskFromJson(
        value,
        mwsf::Type::Entrance,
        descr.toDescription(),
        std::move(attrs)
    );
}

mwsf::TaskNew parkingNewTaskFromJson(const json::Value& value)
{
    mwsf::NoNeededParkingTypeDescr descr;
    if (value.hasField(mwsfa::PARKING_TOLL)) {
        descr.toll = value[mwsfa::PARKING_TOLL].as<bool>();
        descr.changeExistedObject = value.hasField(jsids::OBJECT_ID);
    } else {
        descr.changeExistedObject = false;
    }
    return specificNewTaskFromJson(
        value,
        mwsf::Type::Parking,
        descr.toDescription());
}

mwsf::TaskNew parkingFtTypeNewTaskFromJson(const json::Value& value)
{
    // extract parameters
    int currentTypeAsInt = value[mwsfa::CURRENT_FT_TYPE].as<int>();
    int correctTypeAsInt = value[mwsfa::CORRECT_FT_TYPE].as<int>();

    FtType currentType = static_cast<FtType>(currentTypeAsInt);
    FtType correctType = static_cast<FtType>(correctTypeAsInt);

    static const std::set<FtType> LINEAR_PARKING_TYPES {
        FtType::UrbanRoadnetParkingFree,
        FtType::UrbanRoadnetParkingToll,
        FtType::UrbanRoadnetParkingRestricted,
        FtType::UrbanRoadnetParkingProhibited
    };

    REQUIRE(
        LINEAR_PARKING_TYPES.count(currentType),
        "Invalid current parking ft_type " << currentTypeAsInt
    );
    REQUIRE(
        LINEAR_PARKING_TYPES.count(correctType),
        "Invalid current parking ft_type " << correctTypeAsInt
    );

    // construct attributes
    mwsf::Attrs attrs;
    attrs.addCustom(mwsfa::CURRENT_FT_TYPE, std::to_string(currentTypeAsInt));
    attrs.addCustom(mwsfa::CORRECT_FT_TYPE, std::to_string(correctTypeAsInt));

    // construct description
    mwsf::WrongParkingTypeDescr descr{currentType, correctType};

    return specificNewTaskFromJson(
        value,
        mwsf::Type::Parking,
        descr.toDescription(),
        std::move(attrs)
    );
}

mwsf::TaskNew trafficSignNewTaskFromJson(const json::Value& value)
{
    const auto feedbackType = boost::lexical_cast<mwsf::Type>(
        value[jsids::FEEDBACK_TYPE].as<std::string>()
    );

    mwsf::Description descr;

    switch (feedbackType) {
        case mwsf::Type::MandatoryDirectionTrafficSign:
            descr = mwsf::MandatoryDirectionTrafficSignDescr::toDescription();
            break;
        case mwsf::Type::ProhibitedTurnSign:
            descr = mwsf::ProhibitedTurnSignDescr::toDescription();
            break;
        case mwsf::Type::OneWayTrafficSign:
            descr = mwsf::OneWayTrafficSignDescr::toDescription();
            break;
        case mwsf::Type::TrafficLaneSign:
            descr = mwsf::TrafficLaneSignDescr::toDescription();
            break;
        case mwsf::Type::TrafficProhibitedSign:
            descr = mwsf::TrafficProhibitedSignDescr::toDescription();
            break;
        case mwsf::Type::TrafficCircleSign:
            descr = mwsf::TrafficCircleSignDescr::toDescription();
            break;
        case mwsf::Type::TrucksProhibitedSign:
            descr = mwsf::DescriptionI18n(mwsf::tanker::fb_desc::TRUCKS_PROHIBITED_SIGN_KEY, {});
            break;
        case mwsf::Type::TrucksManeuverRestrictionSign:
            descr = mwsf::DescriptionI18n(mwsf::tanker::fb_desc::TRUCKS_MANEUVER_RESTRICTION_SIGN_KEY, {});
            break;
        case mwsf::Type::TrucksSpeedLimitSign:
            descr = mwsf::DescriptionI18n(mwsf::tanker::fb_desc::TRUCKS_SPEED_LIMIT_SIGN_KEY, {});
            break;
        case mwsf::Type::WeightLimitingSign:
            descr = mwsf::DescriptionI18n(mwsf::tanker::fb_desc::WEIGHT_LIMITING_SIGN_KEY, {});
            break;
        case mwsf::Type::DimensionsLimitingSign:
            descr = mwsf::DescriptionI18n(mwsf::tanker::fb_desc::DIMENSIONS_LIMITING_SIGN_KEY, {});
            break;
        default:
            throw maps::RuntimeError() << "Invalid feedback type " << feedbackType;
    }

    return specificNewTaskFromJson(
        value,
        feedbackType,
        descr
    );
}

social::feedback::TaskNew trafficSignDisappearanceNewTaskFromJson(const json::Value& value)
{
    const auto feedbackType = boost::lexical_cast<mwsf::Type>(
        value[jsids::FEEDBACK_TYPE].as<std::string>()
    );

    mwsf::Description descr;

    switch (feedbackType) {
        case mwsf::Type::ConstructiveSignDisappearance:
            descr = mwsf::DescriptionI18n(mwsf::tanker::fb_desc::CONSTRUCTIVE_SIGN_DISAPPEARANCE_KEY, {});
            break;
        case mwsf::Type::AutomotiveSignDisappearance:
            descr = mwsf::DescriptionI18n(mwsf::tanker::fb_desc::AUTOMOTIVE_SIGN_DISAPPEARANCE_KEY, {});
            break;
        case mwsf::Type::PedestrianSignDisappearance:
            descr = mwsf::DescriptionI18n(mwsf::tanker::fb_desc::PEDESTRIAN_SIGN_DISAPPEARANCE_KEY, {});
            break;
        case mwsf::Type::HeavyVehicleSignDisappearance:
            descr = mwsf::DescriptionI18n(mwsf::tanker::fb_desc::HEAVY_VEHICLE_SIGN_DISAPPEARANCE_KEY, {});
            break;
        case mwsf::Type::SpeedLimitSignDisappearance:
            descr = mwsf::DescriptionI18n(mwsf::tanker::fb_desc::SPEED_LIMIT_SIGN_DISAPPEARANCE_KEY, {});
            break;
        case mwsf::Type::ParkingSignDisappearance:
            descr = mwsf::DescriptionI18n(mwsf::tanker::fb_desc::PARKING_SIGN_DISAPPEARANCE_KEY, {});
            break;
        case mwsf::Type::PublicTransportSignDisappearance:
            descr = mwsf::DescriptionI18n(mwsf::tanker::fb_desc::PUBLIC_TRANSPORT_SIGN_DISAPPEARANCE_KEY, {});
            break;
        case mwsf::Type::TrafficLanesSignDisappearance:
            descr = mwsf::DescriptionI18n(mwsf::tanker::fb_desc::TRAFFIC_LANES_SIGN_DISAPPEARANCE_KEY, {});
            break;
        default:
            throw maps::RuntimeError() << "Invalid feedback type " << feedbackType;
    }

    return specificNewTaskFromJson(
        value,
        feedbackType,
        descr
    );
}

mwsf::TaskNew prohibitedPathNewTaskFromJson(const json::Value& value)
{
    std::optional<mwsf::Movement> movement;
    if (value.hasField(jsids::MOVEMENT)) {
        movement = boost::lexical_cast<mwsf::Movement>(value[jsids::MOVEMENT].toString());
    }
    const auto path = constructProhibitedPathDescrLink(
        geolib3::readGeojson<geolib3::Polyline2>(value[jsids::PATH])
    );

    maps::ymapsdf::rd::AccessId accessId{maps::ymapsdf::rd::AccessId::All};
    if (value.hasField(mwsfa::ACCESS_ID)) {
        accessId =  static_cast<maps::ymapsdf::rd::AccessId>(value[mwsfa::ACCESS_ID].as<int>());
    }

    bool prohibited = true;
    if (value.hasField(mwsfa::COND_TYPE)) {
        prohibited = ("prohibited" == value[mwsfa::COND_TYPE].as<std::string>());
    }
    mwsf::ProhibitedPathDescr descr{movement, path, prohibited, accessId};

    return specificNewTaskFromJson(
        value,
        mwsf::Type::Maneuver,
        descr.toDescription()
    );
}

mwsf::TaskNew onFootNewTaskFromJson(
    DbPools& dbPools,
    types::TUid uid,
    const json::Value& value)
{
    const auto type = value.hasField(jsids::TYPE)
        ? boost::lexical_cast<mwsf::Type>(value[jsids::TYPE].as<std::string>())
        : mwsf::Type::Other;

    mwsf::Attrs attrs;
    std::optional<std::string> userComment;
    if (value.hasField(mwsfa::USER_COMMENT)) {
        userComment = value[mwsfa::USER_COMMENT].as<std::string>();
        attrs.addCustom(mwsfa::USER_DATA_COMMENT, *userComment);
    }

    auto newTask = specificNewTaskFromJson(
        value,
        type,
        sf::DescriptionI18n(sf::tanker::fb_desc::FIX_BY_PHOTO_KEY),
        std::move(attrs)
    );
    if (newTask.source == "experiment-onfoot") {
        static const std::vector<std::string> SPECIAL_ROLE_SOURCES = {
            "sprav-pedestrian-onfoot",
            "lavka-pedestrian-onfoot",
            "partner-pedestrian-onfoot"
        };

        auto coreTxnHandle = dbPools.coreReadTxn();
        const auto roles = userRoles(*coreTxnHandle, uid);
        for (const auto& roleSource : SPECIAL_ROLE_SOURCES) {
            if (roles.contains(roleSource)) {
                newTask.source = roleSource;
                newTask.hidden = true;
                break;
            }
        }
    }
    return newTask;
}

mwsf::TaskNew rootGapNewTaskFromJson(const json::Value& value)
{
    auto bbox = bboxGeoFromRequest(value[jsids::BBOX].as<std::string>());
    auto points = geolib3::readGeojson<geolib3::PointsVector>(value[jsids::POINTS]);
    auto polylines = geolib3::readGeojson<geolib3::PolylinesVector>(value[jsids::POLYLINES]);

    mwsf::RouteGapDescr descr{
        value[jsids::TRACK_COUNT].as<int>(),
        value[jsids::PROBABILITY].as<double>(),
        value[jsids::EXPECTED_TRACK_RIPS].as<double>(),
        constructRouteGapDescrLink(
            bbox,
            points,
            polylines
        )
    };

    return specificNewTaskFromJson(
        value,
        mwsf::Type::RouteGap,
        descr.toDescription()
    );
}

mwsf::TaskNew absentAddressNewTaskFromJson(const json::Value& value)
{
    std::string number = value[mwsfa::ADDRESS_HOUSE_NUMBER].as<std::string>();

    mwsf::Attrs attrs;
    attrs.addCustom(mwsfa::ADDRESS_HOUSE_NUMBER, number);

    mwsf::AbsentAddressDescr descr{ number };

    return specificNewTaskFromJson(
        value,
        mwsf::Type::Address,
        descr.toDescription(),
        std::move(attrs)
    );
}

mwsf::TaskNew absentTrafficLightNewTaskFromJson(const json::Value& value)
{
    return specificNewTaskFromJson(
        value,
        mwsf::Type::TrafficLight,
        mwsf::AbsentTrafficLightDescr::toDescription()
    );
}

mwsf::TaskNew speedBumpSignNewTaskFromJson(const json::Value& value)
{
    ASSERT(value.hasField(mwsfa::SOURCE_CONTEXT));

    mwsf::SpeedBumpSignDescr descr;
    descr.ahead = value[mwsfa::SPEED_BUMP_AHEAD].as<bool>();

    return specificNewTaskFromJson(
        value,
        mwsf::Type::SpeedBumpSign,
        descr.toDescription());
}

namespace {

const std::set<std::string> LANDING_SOURCES {
    "landing", "landing-navi", "landing-taxi"
};

bool isAnyLandingSource(const std::string& source)
{
    return LANDING_SOURCES.count(source);
}

bool isLegacyLandingRequest(const json::Value& body)
{
    std::string type   = body[jsids::TYPE].as<std::string>();
    std::string source = body[jsids::SOURCE].as<std::string>();
    return isAnyLandingSource(source) &&
           type == mwsf::back_compat::ABSENT_OBJECT_TYPE;
}

/*
   Hack due to legacy reasons.
   There are clients who post 'barrier' and 'entrace' feedback
   with 'absent-object' type (which is deprecated),
   and there is no chance to force them to use new handles.
   Here we artificially separate such feedback on several flows,
   deducing type by description parsing
*/
mwsf::TaskNew legacyLandingNewTaskFromJson(const json::Value& value)
{
    std::string descr = value[jsids::DESCRIPTION].as<std::string>();

    auto type = mwsf::back_compat::typeFromDescr(descr);
    REQUIRE(type, "Unable to deduce type from " << descr);

    mwsf::Attrs attrs;
    {
        auto entrance = mwsf::back_compat::extractEntranceName(descr);
        if (entrance) {
            attrs.addCustom(mwsfa::ENTRANCE_NAME, *entrance);
        }
    }

    return specificNewTaskFromJson(
        value,
        *type,
        mwsf::back_compat::deduceFbDescr(descr),
        std::move(attrs));
}

mwsf::TaskNew newTaskFromJson(const json::Value& value)
{
    mwsf::TaskNew newTask(
        geolib3::convertGeodeticToMercator(
            geolib3::readGeojson<geolib3::Point2>(value[jsids::POSITION])
        ),
        boost::lexical_cast<mwsf::Type>(value[jsids::TYPE].as<std::string>()),
        value[jsids::SOURCE].as<std::string>(),
        mwsf::fromJson<mwsf::Description>(value[jsids::DESCRIPTION])
    );

    if (value.hasField(jsids::HIDDEN)) {
        newTask.hidden = value[jsids::HIDDEN].as<bool>();
    }
    if (value.hasField(jsids::INTERNAL_CONTENT)) {
        newTask.internalContent = value[jsids::INTERNAL_CONTENT].as<bool>();
    }

    if (value.hasField(mwsfa::OBJECT_DIFF)) {
        newTask.attrs.add(
            mwsf::AttrType::ObjectDiff, value[mwsfa::OBJECT_DIFF]);
    }

    if (value.hasField(mwsfa::SOURCE_CONTEXT)) {
        newTask.attrs.add(
            mwsf::AttrType::SourceContext, value[mwsfa::SOURCE_CONTEXT]);
    }

    if (value.hasField(jsids::USER_ATTRS)) {
        newTask.attrs.add(mwsf::AttrType::UserData, value[jsids::USER_ATTRS]);
    }

    if (value.hasField(jsids::INDOOR_LEVEL)) {
        newTask.indoorLevel = value[jsids::INDOOR_LEVEL].as<std::string>();
    }

    return newTask;
}

} // namespace

social::feedback::TaskNew createNewTaskFromJson(const json::Value& value)
{
    return isLegacyLandingRequest(value)
        ? legacyLandingNewTaskFromJson(value)
        : newTaskFromJson(value);
}

} // namespace maps::wiki::socialsrv
