#include <maps/wikimap/mapspro/services/social/src/libs/feedback-actions/tasks_helpers.h>

#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <maps/wikimap/mapspro/libs/acl_utils/include/feedback.h>
#include <maps/wikimap/mapspro/services/social/src/libs/feedback/common.h>
#include <maps/wikimap/mapspro/services/social/src/libs/feedback/operations.h>
#include <maps/wikimap/mapspro/services/social/src/libs/feedback/user_processing_level.h>
#include <yandex/maps/wiki/common/robot.h>
#include <yandex/maps/wiki/social/fbapi_issue.h>
#include <yandex/maps/wiki/social/feedback/attribute_names.h>

namespace maps::wiki::socialsrv {

namespace mwsf = social::feedback;

void requireRevealed(const mwsf::Task& task)
{
    SOCIAL_ASSERT(task.revealed(), NotFound);
}

mwsf::TaskForUpdate getTaskForUpdate(
    mwsf::Agent& agent,
    social::TId taskId)
{
    auto task = agent.taskForUpdateById(taskId);
    requireExists(task);
    return std::move(*task);
}

serialize::TaskExtended makeTaskExtended(
    std::optional<mwsf::Task> task,
    const acl_utils::FeedbackChecker& feedbackChecker,
    pqxx::transaction_base& coreTxn,
    pqxx::transaction_base& socialTxn,
    social::TUid uid)
{
    requireExists(task);
    mwsf::GatewayRO fbGw{socialTxn};

    auto history = fbGw.history(task->id());
    auto validOperations = feedbackChecker.restrictValidOperations(
        getTaskValidOperations(*task, socialTxn, uid, coreTxn),
        *task,
        uid);

    auto comments = loadRelativeComments(socialTxn, history.items());
    serialize::TaskExtended result(
        std::move(*task),
        std::move(history),
        getSubstitutionStrategy(feedbackChecker.aclChecker(), uid),
        std::move(comments),
        std::move(validOperations),
        uid);

    if (feedbackChecker.aclChecker().userHasPermission(uid, TASK_FBAPI_ISSUE_ID_PERMISSION)) {
        const auto issues = social::fbapiIssuesByFilter(
            socialTxn,
            social::FbapiIssueFilter().feedbackTaskId(result.taskForUI.id()));
        if (!issues.empty()) {
            ASSERT(issues.size() == 1);
            result.fbapiIssueId = issues.front().issueId();
        }
    }

    return result;
}

void checkPermissionForTask(
    const acl_utils::FeedbackChecker& feedbackChecker,
    social::TUid uid,
    const mwsf::Task& task,
    const mwsf::History& history,
    mwsf::TaskOperation operation)
{
    auto& aclChecker = feedbackChecker.aclChecker();
    bool checkNeedInfo = true;
    switch (operation) {
        case mwsf::TaskOperation::Reject:
        case mwsf::TaskOperation::Accept:
        case mwsf::TaskOperation::Open:
        case mwsf::TaskOperation::Acquire:
            aclChecker.checkPermission(uid, RESOLVE_TASK_PERMISSION);
            break;

        case mwsf::TaskOperation::NeedInfo:
            aclChecker.checkPermission(uid, NEED_INFO_TASK_PERMISSION);
            break;

        case mwsf::TaskOperation::ChangeType:
            aclChecker.checkPermission(uid, CHANGE_TYPE_TASK_PERMISSION);
            break;

        case mwsf::TaskOperation::ChangePosition:
            aclChecker.checkPermission(uid, CHANGE_POSITION_TASK_PERMISSION);
            break;

        case mwsf::TaskOperation::Hide:
        case mwsf::TaskOperation::Show:
            aclChecker.checkPermission(uid, HIDDEN_TASK_PERMISSION);
            break;

        case mwsf::TaskOperation::Release:
            return;  // always acceptable

        case mwsf::TaskOperation::MarkViewed:
            checkNeedInfo = false;
            break;

        case mwsf::TaskOperation::Create:
        case mwsf::TaskOperation::Defer:
        case mwsf::TaskOperation::Reveal:
        case mwsf::TaskOperation::Deploy:
        case mwsf::TaskOperation::Comment:
        case mwsf::TaskOperation::ProcessingLevelUp:
        case mwsf::TaskOperation::ProcessingLevelDown:
        case mwsf::TaskOperation::ChangeProcessingLvl:
        case mwsf::TaskOperation::Unrecognized:
            throw yacare::errors::InternalError() << "Invalid operation: " << operation;
    }

    if (task.processingLevel() > 0) {
        if (operation == mwsf::TaskOperation::Open &&
            serialize::isJustLeveledUpByUser(history, uid)) {
            // no check
        } else if (operation == mwsf::TaskOperation::MarkViewed) {
            // no check
        } else {
            aclChecker.checkPermission(uid, PROCESSING_LVL1_PERMISSION);
        }
    }
    SOCIAL_REQUIRE(
        feedbackChecker.isAllowedToModify(task, uid),
        Forbidden,
        "Access to feedback " << task.id() << " forbidden for user " << uid);

    if (checkNeedInfo && task.state() == mwsf::TaskState::NeedInfo) {
        aclChecker.checkPermission(uid, NEED_INFO_TASK_PERMISSION);
    }
}

PostTaskResult postNewTask(
    DbPools& dbPools,
    const mwsf::TaskNew& newTask)
{
    auto writeContext = dbPools.writeContext();
    auto uid = common::ROBOT_UID;
    mwsf::Agent agent(writeContext.socialTxn(), uid);

    auto addedTask = agent.addTask(newTask);
    auto history = agent.gatewayRo().history(addedTask.id());
    social::Comments comments; // no comments in just created Task

    auto token = writeContext.commit();

    return PostTaskResult{
        serialize::TaskExtended(
            std::move(addedTask),
            std::move(history),
            serialize::SubstitutionStrategy::None,
            std::move(comments),
            mwsf::TaskOperations{}, // created task unavailable for UI
            uid),
        token};
}

} // namespace maps::wiki::socialsrv
