#include <library/cpp/testing/unittest/env.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/libs/acl/impl/factory.h>
#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <maps/wikimap/mapspro/libs/acl/include/exception.h>
#include <maps/wikimap/mapspro/libs/revisionapi/include/yandex/maps/wiki/revisionapi/revisionapi.h>
#include <maps/wikimap/mapspro/libs/social/include/yandex/maps/wiki/social/feedback/agent.h>
#include <maps/wikimap/mapspro/libs/social/include/yandex/maps/wiki/social/feedback/preset.h>
#include <maps/wikimap/mapspro/libs/social/include/yandex/maps/wiki/social/feedback/task_feed.h>
#include <maps/wikimap/mapspro/libs/social/tests/helpers/fb_task_creator.h>
#include <maps/wikimap/mapspro/libs/social/tests/helpers/fb_task_factory.h>
#include <maps/wikimap/mapspro/services/social/src/libs/feedback/serialize.h>
#include <maps/wikimap/mapspro/services/social/src/libs/feedback-actions/presets.h>
#include <maps/wikimap/mapspro/services/social/src/libs/feedback-actions/tasks_helpers.h>
#include <maps/wikimap/mapspro/services/social/src/libs/test_helpers/helpers.h>
#include <fstream>

template <>
void Out<maps::wiki::social::feedback::Preset>(
    IOutputStream& os,
    const maps::wiki::social::feedback::Preset& preset)
{
    std::ostringstream ostr;
    ostr << preset;
    os << ostr.str();
}

namespace maps::wiki::socialsrv::tests {

using social::feedback::tests::FbTaskCreator;
namespace sf = social::feedback;

namespace {

const social::TId PRESET_ID = 1;

const auto JSON_PRESET = maps::json::Value::fromString(R"(
    {
        "id": "474747",
        "name": "my_favourite_preset",
        "hidden": false,
        "status": "opened"
    }
)");

const auto JSON_PRESET_RESOLVED = maps::json::Value::fromString(R"(
    {
        "id": "474747",
        "name": "my_favourite_preset",
        "hidden": false,
        "status": "resolved",
        "sources": ["fbapi-samsara"]
    }
)");

sf::TaskForUpdate moveFeedbackToAoi(
    sf::Agent& agent,
    const sf::TaskForUpdate& task,
    ID aoiId)
{
    auto acquiredTask = agent.acquireTask(task);
    UNIT_ASSERT(acquiredTask);
    auto changedTask = agent.changeTaskPosition(*acquiredTask, task.position(), {aoiId});
    UNIT_ASSERT(changedTask);
    return std::move(*changedTask);
}

sf::Preset getPresetMandatory(pqxx::transaction_base& socialTxn, social::TId presetId)
{
    auto preset = sf::getPreset(socialTxn, presetId);
    UNIT_ASSERT(preset);
    return std::move(*preset);
}

social::TId createAoi(pgpool3::Pool& pool)
{
    static const acl::UID USER_ID = 101;
    static const social::TId AOI_ID = 2001;
    const std::string AOI_DATA_PATH = SRC_("data/aoi.json");
    {   // import data from json
        std::ifstream json(AOI_DATA_PATH);
        UNIT_ASSERT_C(json, "Could not open file " << AOI_DATA_PATH);

        revisionapi::RevisionAPI revision(pool);
        revision.importData(USER_ID, revisionapi::IdMode::StartFromJsonId, json);
    }
    auto txn = pool.masterWriteableTransaction();
    txn->exec("ALTER SEQUENCE revision.object_id_seq RESTART WITH 13787134532");
    txn->commit();
    return AOI_ID;
}

sf::PresetRoles makeRoles(social::TId readRoleId, social::TId allRightsRoleId)
{
    sf::PresetRoles roles;
    roles.setId(sf::RoleKind::Read, readRoleId);
    roles.setId(sf::RoleKind::AllRights, allRightsRoleId);
    return roles;
}

}  // unnamed namespace

Y_UNIT_TEST_SUITE(feedback_presets_tests) {

Y_UNIT_TEST_F(createPreset, DbFixture) {
    const auto jsonPresetResult = maps::json::Value::fromString(
        censorToken(createPreset(JSON_PRESET, dbPools().writeContext()))
    );

    UNIT_ASSERT_VALUES_EQUAL(
        jsonPresetResult,
        maps::json::Value::fromString(R"(
        {
            "token": "123:1:core.321:2:social",
            "feedbackPreset": {
                "id": "1",
                "name": "my_favourite_preset",
                "hidden":false,
                "status": "opened"
            }
        }
        )")
    );

    sf::Preset preset{
        .id = std::stoull(jsonPresetResult["feedbackPreset"]["id"].as<std::string>()),
        .name = jsonPresetResult["feedbackPreset"]["name"].as<std::string>(),
        .roles = makeRoles(1, 2),
        .entries = internal::presetEntriesFromJson(jsonPresetResult["feedbackPreset"])
    };

    auto presetFromDb = sf::getPreset(*dbPools().socialReadTxn(), PRESET_ID);
    UNIT_ASSERT(presetFromDb);
    UNIT_ASSERT_VALUES_EQUAL(*presetFromDb, preset);

    auto coreReadTxn = dbPools().coreReadTxn();
    acl::ACLGateway aclGateway(*coreReadTxn);
    {
        auto role = aclGateway.role(presetFromDb->roles.getId(sf::RoleKind::Read));
        UNIT_ASSERT_VALUES_EQUAL(role.name(), "feedbackPreset_my_favourite_preset_Read");
        UNIT_ASSERT_VALUES_EQUAL(
            role.description(),
            "Autogenerated role. Do not edit/delete it manually. "
            "Read role for feedback preset 'my_favourite_preset'."
        );
        UNIT_ASSERT(role.permissions().empty());
    }
    {
        auto role = aclGateway.role(presetFromDb->roles.getId(sf::RoleKind::AllRights));
        UNIT_ASSERT_VALUES_EQUAL(role.name(), "feedbackPreset_my_favourite_preset_AllRights");
        UNIT_ASSERT_VALUES_EQUAL(
            role.description(),
            "Autogenerated role. Do not edit/delete it manually. "
            "AllRights role for feedback preset 'my_favourite_preset'."
        );
        UNIT_ASSERT(role.permissions().empty());
    }
}

Y_UNIT_TEST_F(updatePreset, DbFixture) {
    createPreset(JSON_PRESET, dbPools().writeContext());

    const auto updateJsonPreset = maps::json::Value::fromString(R"(
        {
            "name": "updated_name",
            "hidden": true
        }
    )");
    const auto jsonPresetResult = maps::json::Value::fromString(
        // presetId = 1
        censorToken(updatePreset(1, updateJsonPreset, dbPools().writeContext()))
    );

    UNIT_ASSERT_VALUES_EQUAL(
        jsonPresetResult,
        maps::json::Value::fromString(R"(
        {
            "token": "123:1:core.321:2:social",
            "feedbackPreset": {
                "id": "1",
                "name": "updated_name",
                "hidden": true
            }
        }
        )")
    );

    sf::Preset preset{
        .id = 1,
        .name = "updated_name",
        .roles = makeRoles(1, 2),
        .entries = {
            .hidden = true
        }
    };

    auto presetFromDb = sf::getPreset(*dbPools().socialReadTxn(), PRESET_ID);
    UNIT_ASSERT(presetFromDb);
    UNIT_ASSERT_VALUES_EQUAL(*presetFromDb, preset);

    auto coreReadTxn = dbPools().coreReadTxn();
    acl::ACLGateway aclGateway(*coreReadTxn);
    {
        auto role = aclGateway.role(presetFromDb->roles.getId(sf::RoleKind::Read));
        UNIT_ASSERT_VALUES_EQUAL(role.name(), "feedbackPreset_updated_name_Read");
        UNIT_ASSERT_VALUES_EQUAL(
            role.description(),
            "Autogenerated role. Do not edit/delete it manually. "
            "Read role for feedback preset 'updated_name'."
        );
        UNIT_ASSERT(role.permissions().empty());
    }
    {
        auto role = aclGateway.role(presetFromDb->roles.getId(sf::RoleKind::AllRights));
        UNIT_ASSERT_VALUES_EQUAL(role.name(), "feedbackPreset_updated_name_AllRights");
        UNIT_ASSERT_VALUES_EQUAL(
            role.description(),
            "Autogenerated role. Do not edit/delete it manually. "
            "AllRights role for feedback preset 'updated_name'."
        );
        UNIT_ASSERT(role.permissions().empty());
    }
}

Y_UNIT_TEST_F(deletePreset, DbFixture) {
    UNIT_ASSERT_NO_EXCEPTION(deletePreset(1, dbPools().writeContext()));

    createPreset(JSON_PRESET, dbPools().writeContext());

    auto coreReadTxn = dbPools().coreReadTxn();
    acl::ACLGateway aclGateway(*coreReadTxn);

    auto socialReadTxn = dbPools().socialReadTxn();
    UNIT_ASSERT(sf::getPreset(*socialReadTxn, PRESET_ID));
    UNIT_ASSERT_NO_EXCEPTION(aclGateway.role("feedbackPreset_my_favourite_preset_Read"));
    UNIT_ASSERT_NO_EXCEPTION(aclGateway.role("feedbackPreset_my_favourite_preset_AllRights"));

    UNIT_ASSERT_NO_EXCEPTION(deletePreset(1, dbPools().writeContext()));
    UNIT_ASSERT(!sf::getPreset(*socialReadTxn, PRESET_ID));
    UNIT_ASSERT_EXCEPTION(
        aclGateway.role("feedbackPreset_my_favourite_preset_Read"),
        acl::RoleNotExists
    );
    UNIT_ASSERT_EXCEPTION(
        aclGateway.role("feedbackPreset_my_favourite_preset_AllRights"),
        acl::RoleNotExists
    );
}

Y_UNIT_TEST_F(assignedPresets, DbFixture) {
    // create preset
    createPreset(JSON_PRESET, dbPools().writeContext());

    // create user 1
    static const acl::UID USER_ID = 101;
    static const std::string USER_LOGIN = "Vasya.Pupkin";
    {
        auto writeContext = dbPools().writeContext();
        acl::ACLGateway aclGateway(writeContext.coreTxn());
        aclGateway.createUser(USER_ID, USER_LOGIN, USER_LOGIN, USER_ID);
        writeContext.commit();
    }

    // create user 2
    static const acl::UID USER2_ID = 102;
    static const std::string USER2_LOGIN = "Vasya.Pupkin.2";
    {
        auto writeContext = dbPools().writeContext();
        acl::ACLGateway aclGateway(writeContext.coreTxn());
        aclGateway.createUser(USER2_ID, USER2_LOGIN, USER2_LOGIN, USER2_ID);
        writeContext.commit();
    }

    // create aoi
    auto aoiId = createAoi(pool());

    // assign preset-AllRights-role to user 1
    {
        auto writeContext = dbPools().writeContext();
        acl::ACLGateway aclGateway(writeContext.coreTxn());
        auto presetFromDb = sf::getPreset(writeContext.socialTxn(), PRESET_ID);
        auto user = aclGateway.user(USER_ID);
        auto role = aclGateway.role(presetFromDb->roles.getId(sf::RoleKind::AllRights));
        auto aoi = acl::Factory::aoi(aoiId, "NAME", "WKB", acl::Deleted::No);
        aclGateway.createPolicy(user, role, aoi);
        writeContext.commit();
    }

    // assign preset-Read-role to user 2
    {
        auto writeContext = dbPools().writeContext();
        acl::ACLGateway aclGateway(writeContext.coreTxn());
        auto presetFromDb = sf::getPreset(writeContext.socialTxn(), PRESET_ID);
        auto user = aclGateway.user(USER2_ID);
        auto role = aclGateway.role(presetFromDb->roles.getId(sf::RoleKind::Read));
        auto aoi = acl::Factory::aoi(aoiId, "NAME", "WKB", acl::Deleted::No);
        aclGateway.createPolicy(user, role, aoi);
        writeContext.commit();
    }

    // create feedback within aoi
    {
        auto writeContext = dbPools().writeContext();
        auto task = FbTaskCreator(writeContext.socialTxn(), sf::TaskState::Opened)
            .source("fbapi-samsara")
            .create();

        sf::Agent agent(writeContext.socialTxn(), USER_ID);
        moveFeedbackToAoi(agent, task, aoiId);
        writeContext.commit();
    }

    // check user 1
    {
        auto writeContext = dbPools().writeContext();
        auto result = socialsrv::getAssignedPresetsResponse(
            writeContext.coreTxn(),
            writeContext.socialTxn(),
            acl_utils::FeedbackPresetCheckerImpl(pool(), pool(), 10, std::chrono::seconds(10)),
            USER_ID);

        auto resultJson = json::Value::fromString(socialsrv::toJson(result));
        const auto correctJson = json::Value::fromString(R"(
            [{
                "preset": {
                    "id": "1",
                    "name": "my_favourite_preset",
                    "hidden": false,
                    "status": "opened"
                },
                "regions": [{
                    "aoi": {
                        "id": "2001",
                        "title": "NNN"
                    },
                    "totalCount": 1
                }]
            }]
        )");
        UNIT_ASSERT_VALUES_EQUAL(
            (json::Builder() << resultJson).str(),
            (json::Builder() << correctJson).str());
    }

    // check user 2
    // Read role doesn't allow access to assigned presets
    {
        auto writeContext = dbPools().writeContext();
        auto result = socialsrv::getAssignedPresetsResponse(
            writeContext.coreTxn(),
            writeContext.socialTxn(),
            acl_utils::FeedbackPresetCheckerImpl(pool(), pool(), 10, std::chrono::seconds(10)),
            USER2_ID);

        auto resultJson = json::Value::fromString(socialsrv::toJson(result));
        const auto correctJson = json::Value::fromString("[]");
        UNIT_ASSERT_VALUES_EQUAL(
            (json::Builder() << resultJson).str(),
            (json::Builder() << correctJson).str());
    }
}

Y_UNIT_TEST_F(presetTasksFeed, DbFixture) {
    static const ID AOI_ID = 30;
    static const ID AOI_ID_OTHER = 40;

    // create preset
    // status (==resolved) should be ignored in this test
    createPreset(JSON_PRESET_RESOLVED, dbPools().writeContext());

    // create user
    static const acl::UID USER_ID = 101;
    static const std::string USER_LOGIN = "Vasya.Pupkin";
    {
        auto writeContext = dbPools().writeContext();
        acl::ACLGateway aclGateway(writeContext.coreTxn());
        aclGateway.createUser(USER_ID, USER_LOGIN, USER_LOGIN, USER_ID);
        writeContext.commit();
    }

    // create user 2
    static const acl::UID USER2_ID = 102;
    static const std::string USER2_LOGIN = "Vasya.Pupkin.2";
    {
        auto writeContext = dbPools().writeContext();
        acl::ACLGateway aclGateway(writeContext.coreTxn());
        aclGateway.createUser(USER2_ID, USER2_LOGIN, USER2_LOGIN, USER2_ID);
        writeContext.commit();
    }

    // assign preset-AllRights-role to user
    {
        auto writeContext = dbPools().writeContext();
        acl::ACLGateway aclGateway(writeContext.coreTxn());
        auto presetFromDb = sf::getPreset(writeContext.socialTxn(), PRESET_ID);
        auto user = aclGateway.user(USER_ID);
        auto role = aclGateway.role(presetFromDb->roles.getId(sf::RoleKind::AllRights));
        auto aoi = aclGateway.aoi(AOI_ID);
        aclGateway.createPolicy(user, role, aoi);
        writeContext.commit();
    }

    // assign preset-Read-role to user 2
    {
        auto writeContext = dbPools().writeContext();
        acl::ACLGateway aclGateway(writeContext.coreTxn());
        auto presetFromDb = sf::getPreset(writeContext.socialTxn(), PRESET_ID);
        auto user = aclGateway.user(USER2_ID);
        auto role = aclGateway.role(presetFromDb->roles.getId(sf::RoleKind::Read));
        auto aoi = aclGateway.aoi(AOI_ID);
        aclGateway.createPolicy(user, role, aoi);
        writeContext.commit();
    }

    acl_utils::CachingAclChecker aclChecker(pool(),
        10, std::chrono::seconds(0));

    // create feedback NOT within preset by filter
    {
        auto writeContext = dbPools().writeContext();
        auto task = FbTaskCreator(writeContext.socialTxn(), sf::TaskState::Opened)
            .source("geoq-newsfeed")
            .create();

        sf::Agent agent(writeContext.socialTxn(), USER_ID);
        moveFeedbackToAoi(agent, task, AOI_ID);
        writeContext.commit();
    }

    // check request. No feedback by filter
    {
        auto writeContext = dbPools().writeContext();

        auto result = socialsrv::getPresetsTasksFeed(
            writeContext.socialTxn(),
            acl_utils::FeedbackChecker(
                aclChecker,
                acl_utils::FeedbackPresetCheckerImpl(pool(), pool(), 10, std::chrono::seconds(10))
            ),
            USER_ID,
            PRESET_ID,
            AOI_ID,
            sf::TaskFeedParamsId(0, 0, 1, social::TasksOrder::OldestFirst));

        UNIT_ASSERT(result.hasMore == social::HasMore::No);
        UNIT_ASSERT(result.totalCount == 0);
    }

    // create feedback NOT within preset by aoi
    {
        auto writeContext = dbPools().writeContext();
        auto task = FbTaskCreator(writeContext.socialTxn(), sf::TaskState::Opened)
            .source("fbapi-samsara")
            .create();

        sf::Agent agent(writeContext.socialTxn(), USER_ID);
        moveFeedbackToAoi(agent, task, AOI_ID_OTHER);
        writeContext.commit();
    }

    // check request. No feedback by aoi
    {
        auto writeContext = dbPools().writeContext();

        auto result = socialsrv::getPresetsTasksFeed(
            writeContext.socialTxn(),
            acl_utils::FeedbackChecker(
                aclChecker,
                acl_utils::FeedbackPresetCheckerImpl(pool(), pool(), 10, std::chrono::seconds(10))
            ),
            USER_ID,
            PRESET_ID,
            AOI_ID,
            sf::TaskFeedParamsId(0, 0, 1, social::TasksOrder::OldestFirst));

        UNIT_ASSERT(result.hasMore == social::HasMore::No);
        UNIT_ASSERT(result.totalCount == 0);
    }

    // create feedback within assigned preset
    social::TId taskInFeedId0;
    {
        auto writeContext = dbPools().writeContext();
        auto task = FbTaskCreator(writeContext.socialTxn(), sf::TaskState::Opened)
            .source("fbapi-samsara")
            .create();
        taskInFeedId0 = task.id();

        sf::Agent agent(writeContext.socialTxn(), USER_ID);
        moveFeedbackToAoi(agent, task, AOI_ID);
        writeContext.commit();
    }

    // create feedback within assigned preset
    social::TId taskInFeedId1;
    {
        auto writeContext = dbPools().writeContext();
        auto task = FbTaskCreator(writeContext.socialTxn(), sf::TaskState::Opened)
            .source("fbapi-samsara")
            .create();
        taskInFeedId1 = task.id();

        sf::Agent agent(writeContext.socialTxn(), USER_ID);
        moveFeedbackToAoi(agent, task, AOI_ID);
        writeContext.commit();
    }

    // create feedback within assigned preset (resolved one)
    {
        auto writeContext = dbPools().writeContext();
        auto task = FbTaskCreator(writeContext.socialTxn(), sf::TaskState::Opened)
            .source("fbapi-samsara")
            .create();

        sf::Agent agent(writeContext.socialTxn(), USER_ID);
        auto taskInAoi = moveFeedbackToAoi(agent, task, AOI_ID);

        UNIT_ASSERT(agent.resolveTaskCascade(taskInAoi, sf::Resolution::createAccepted(), std::nullopt));
        writeContext.commit();
    }

    // check request for user 2
    // Read role doesn't allow access to preset tasks feed
    {
        auto writeContext = dbPools().writeContext();

        UNIT_ASSERT_EXCEPTION(
            socialsrv::getPresetsTasksFeed(
                writeContext.socialTxn(),
                acl_utils::FeedbackChecker(
                    aclChecker,
                    acl_utils::FeedbackPresetCheckerImpl(pool(), pool(), 10, std::chrono::seconds(10))
                ),
                USER2_ID,
                PRESET_ID,
                AOI_ID,
                sf::TaskFeedParamsId(0, 0, 10, social::TasksOrder::OldestFirst)),
            Error
        );
    }

    // check request. one from begining
    {
        auto writeContext = dbPools().writeContext();

        auto result = socialsrv::getPresetsTasksFeed(
            writeContext.socialTxn(),
            acl_utils::FeedbackChecker(
                aclChecker,
                acl_utils::FeedbackPresetCheckerImpl(pool(), pool(), 10, std::chrono::seconds(10))
            ),
            USER_ID,
            PRESET_ID,
            AOI_ID,
            sf::TaskFeedParamsId(0, 0, 1, social::TasksOrder::OldestFirst));

        UNIT_ASSERT(result.hasMore == social::HasMore::Yes);
        UNIT_ASSERT(result.totalCount == 2);

        UNIT_ASSERT_VALUES_EQUAL(result.tasks.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(result.tasks.at(0).id(), taskInFeedId0);
    }

    // check request. 10 after first
    {
        auto writeContext = dbPools().writeContext();

        auto result = socialsrv::getPresetsTasksFeed(
            writeContext.socialTxn(),
            acl_utils::FeedbackChecker(
                aclChecker,
                acl_utils::FeedbackPresetCheckerImpl(pool(), pool(), 10, std::chrono::seconds(10))
            ),
            USER_ID,
            PRESET_ID,
            AOI_ID,
            sf::TaskFeedParamsId(0, taskInFeedId0, 10, social::TasksOrder::OldestFirst));

        UNIT_ASSERT(result.hasMore == social::HasMore::No);
        UNIT_ASSERT(result.totalCount == 2);

        UNIT_ASSERT_VALUES_EQUAL(result.tasks.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(result.tasks.at(0).id(), taskInFeedId1);
    }

    // check request. two bofore second+1
    {
        auto writeContext = dbPools().writeContext();

        auto result = socialsrv::getPresetsTasksFeed(
            writeContext.socialTxn(),
            acl_utils::FeedbackChecker(
                aclChecker,
                acl_utils::FeedbackPresetCheckerImpl(pool(), pool(), 10, std::chrono::seconds(10))
            ),
            USER_ID,
            PRESET_ID,
            AOI_ID,
            sf::TaskFeedParamsId(taskInFeedId1+1, 0, 2, social::TasksOrder::OldestFirst));

        UNIT_ASSERT(result.hasMore == social::HasMore::No);
        UNIT_ASSERT(result.totalCount == 2);

        UNIT_ASSERT_VALUES_EQUAL(result.tasks.size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(result.tasks.at(0).id(), taskInFeedId0);
        UNIT_ASSERT_VALUES_EQUAL(result.tasks.at(1).id(), taskInFeedId1);
    }
}

Y_UNIT_TEST_F(getAssignedPresets, DbFixture) {
    static const acl::UID USER1_ID = 101;
    static const acl::UID USER2_ID = 102;
    static const acl::UID USER3_ID = 103;
    static const std::string USER_LOGIN = "Vasya.Pupkin";
    static const ID AOI_ID = 30;
    static const ID AOI2_ID = 32;
    static const auto JSON_PRESET2 = maps::json::Value::fromString(R"(
    {
        "id": "474747",
        "name": "my_favourite_preset2",
        "hidden": false,
        "status": "opened"
    }
    )");
    static const auto JSON_PRESET3 = maps::json::Value::fromString(R"(
    {
        "id": "474747",
        "name": "my_favourite_preset3",
        "hidden": false,
        "status": "opened"
    }
    )");

    // create presets
    createPreset(JSON_PRESET, dbPools().writeContext());
    static const social::TId PRESET1_ID = 1;
    auto preset1 = getPresetMandatory(*dbPools().socialReadTxn(), PRESET1_ID);

    createPreset(JSON_PRESET2, dbPools().writeContext());
    static const social::TId PRESET2_ID = 2;
    auto preset2 = getPresetMandatory(*dbPools().socialReadTxn(), PRESET2_ID);

    createPreset(JSON_PRESET3, dbPools().writeContext());

    // create users
    {
        auto writeContext = dbPools().writeContext();
        acl::ACLGateway aclGateway(writeContext.coreTxn());
        aclGateway.createUser(USER1_ID, USER_LOGIN + "1", USER_LOGIN, USER1_ID);
        aclGateway.createUser(USER2_ID, USER_LOGIN + "2", USER_LOGIN, USER2_ID);
        aclGateway.createUser(USER3_ID, USER_LOGIN + "3", USER_LOGIN, USER3_ID);
        writeContext.commit();
    }

    // assign preset-role to USER1
    {
        auto writeContext = dbPools().writeContext();
        acl::ACLGateway aclGateway(writeContext.coreTxn());
        auto user = aclGateway.user(USER1_ID);
        auto role = aclGateway.role(preset1.roles.getId(sf::RoleKind::AllRights));
        auto aoi = acl::Factory::aoi(AOI_ID, "NAME", "WKB", acl::Deleted::No);
        aclGateway.createPolicy(user, role, aoi);
        writeContext.commit();
    }

    // assign preset-role to USER2
    {
        auto writeContext = dbPools().writeContext();
        acl::ACLGateway aclGateway(writeContext.coreTxn());
        auto user = aclGateway.user(USER2_ID);
        auto role = aclGateway.role(preset2.roles.getId(sf::RoleKind::AllRights));
        auto aoi = acl::Factory::aoi(AOI2_ID, "NAME2", "WKB", acl::Deleted::No);
        aclGateway.createPolicy(user, role, aoi);
        writeContext.commit();
    }

    // check USER1
    {
        auto writeContext = dbPools().writeContext();
        auto allPresets = sf::getPresets(writeContext.socialTxn());
        UNIT_ASSERT_VALUES_EQUAL(allPresets.size(), 3);
        acl_utils::FeedbackPresetCheckerImpl fbPresetChecker(pool(), pool(), 10, std::chrono::seconds(10));
        auto ap = fbPresetChecker.getAssignedPresets(USER1_ID);

        auto assignedPresets = ap.getAssignedPresets(acl_utils::PresetRightKind::Modify);
        UNIT_ASSERT_VALUES_EQUAL(assignedPresets.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(assignedPresets[0].presetId, PRESET1_ID);
        UNIT_ASSERT(assignedPresets[0].aoiId);
        UNIT_ASSERT_VALUES_EQUAL(assignedPresets[0].aoiId.value(), AOI_ID);

        UNIT_ASSERT(fbPresetChecker.hasModifyPermission(USER1_ID, preset1, AOI_ID));
        UNIT_ASSERT(!fbPresetChecker.hasModifyPermission(USER1_ID, preset1, AOI2_ID));
        UNIT_ASSERT(!fbPresetChecker.hasModifyPermission(USER1_ID, preset2, AOI_ID));
        UNIT_ASSERT(!fbPresetChecker.hasModifyPermission(USER1_ID, preset2, AOI2_ID));
    }

    // check USER2
    {
        auto writeContext = dbPools().writeContext();
        auto allPresets = sf::getPresets(writeContext.socialTxn());
        acl_utils::FeedbackPresetCheckerImpl fbPresetChecker(pool(), pool(), 10, std::chrono::seconds(10));
        auto ap = fbPresetChecker.getAssignedPresets(USER2_ID);

        auto assignedPresets = ap.getAssignedPresets(acl_utils::PresetRightKind::Modify);
        UNIT_ASSERT_VALUES_EQUAL(assignedPresets.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(assignedPresets[0].presetId, PRESET2_ID);
        UNIT_ASSERT(assignedPresets[0].aoiId);
        UNIT_ASSERT_VALUES_EQUAL(assignedPresets[0].aoiId.value(), AOI2_ID);

        UNIT_ASSERT(!fbPresetChecker.hasModifyPermission(USER2_ID, preset1, AOI_ID));
        UNIT_ASSERT(!fbPresetChecker.hasModifyPermission(USER2_ID, preset1, AOI2_ID));
        UNIT_ASSERT(!fbPresetChecker.hasModifyPermission(USER2_ID, preset2, AOI_ID));
        UNIT_ASSERT(fbPresetChecker.hasModifyPermission(USER2_ID, preset2, AOI2_ID));
    }

    // check USER3
    {
        acl_utils::FeedbackPresetCheckerImpl fbPresetChecker(pool(), pool(), 10, std::chrono::seconds(10));
        auto ap = fbPresetChecker.getAssignedPresets(USER3_ID);

        auto assignedPresets = ap.getAssignedPresets(acl_utils::PresetRightKind::Modify);
        UNIT_ASSERT_VALUES_EQUAL(assignedPresets.size(), 0);
    }
}

Y_UNIT_TEST_F(baseDimensions_restrictTaskFilter, DbFixture) {
    static const sf::TaskFeedParamsId FEED_PARAMS(0, 0, 100, social::TasksOrder::OldestFirst);
    static const acl::UID USER1_ID = 101;
    static const acl::UID USER2_ID = 102;
    static const std::string USER_LOGIN = "Vasya.Pupkin";
    static const ID AOI_ID = 30;
    static const auto JSON_PRESET1 = maps::json::Value::fromString(R"(
    {
        "name": "my_favourite_preset1",
        "types": ["subway", "address", "road", "poi"],
        "sources": ["fbapi", "not-feedback-workflow"],
        "workflows": ["feedback"],
        "hidden": false
    }
    )");
    static const auto JSON_PRESET2 = maps::json::Value::fromString(R"(
    {
        "name": "my_favourite_preset2",
        "types": ["address", "fence"],
        "sources": ["fbapi"]
    }
    )");
    static const auto JSON_PRESET3 = maps::json::Value::fromString(R"(
    {
        "name": "my_favourite_preset3",
        "types": ["road", "barrier"]
    }
    )");

    // create presets
    createPreset(JSON_PRESET1, dbPools().writeContext());
    static const social::TId PRESET1_ID = 1;
    auto preset1 = getPresetMandatory(*dbPools().socialReadTxn(), PRESET1_ID);

    createPreset(JSON_PRESET2, dbPools().writeContext());
    static const social::TId PRESET2_ID = 2;
    auto preset2 = getPresetMandatory(*dbPools().socialReadTxn(), PRESET2_ID);

    createPreset(JSON_PRESET3, dbPools().writeContext());
    static const social::TId PRESET3_ID = 3;
    auto preset3 = getPresetMandatory(*dbPools().socialReadTxn(), PRESET3_ID);

    // create users
    {
        auto writeContext = dbPools().writeContext();
        acl::ACLGateway aclGateway(writeContext.coreTxn());
        aclGateway.createUser(USER1_ID, USER_LOGIN + "1", USER_LOGIN, USER1_ID);
        aclGateway.createUser(USER2_ID, USER_LOGIN + "2", USER_LOGIN, USER2_ID);
        writeContext.commit();
    }

    // assign preset-roles to USER1
    {
        auto writeContext = dbPools().writeContext();
        acl::ACLGateway aclGateway(writeContext.coreTxn());
        auto user = aclGateway.user(USER1_ID);
        auto role1 = aclGateway.role(preset1.roles.getId(sf::RoleKind::AllRights));
        auto role2 = aclGateway.role(preset2.roles.getId(sf::RoleKind::AllRights));
        auto role3 = aclGateway.role(preset3.roles.getId(sf::RoleKind::AllRights));
        auto aoi = acl::Factory::aoi(AOI_ID, "NAME", "WKB", acl::Deleted::No);
        auto zeroAoi = acl::Factory::aoi(0, "-", "WKB", acl::Deleted::No);
        aclGateway.createPolicy(user, role1, zeroAoi);
        aclGateway.createPolicy(user, role2, zeroAoi);
        aclGateway.createPolicy(user, role3, aoi);
        writeContext.commit();
    }

    // create feedback within preset1 && preset2
    social::TId taskInPreset1And2 = [&](){
        auto writeContext = dbPools().writeContext();
        auto task = FbTaskCreator(writeContext.socialTxn(), sf::TaskState::Opened)
            .source("fbapi")
            .type(sf::Type::Address)
            .create();
        writeContext.commit();
        INFO() << "ASDF: " << task.id();
        return task.id();
    }();

    // create feedback within preset1 only
    social::TId taskInPreset1 = [&](){
        auto writeContext = dbPools().writeContext();
        auto task = FbTaskCreator(writeContext.socialTxn(), sf::TaskState::Opened)
            .source("fbapi")
            .type(sf::Type::Poi)
            .create();
        writeContext.commit();
        return task.id();
    }();

    // create feedback within preset2 only
    social::TId taskInPreset2 = [&](){
        auto writeContext = dbPools().writeContext();
        auto task = FbTaskCreator(writeContext.socialTxn(), sf::TaskState::Opened)
            .source("fbapi")
            .type(sf::Type::Fence)
            .create();
        writeContext.commit();
        return task.id();
    }();

    // create feedback within preset3 only
    {
        auto writeContext = dbPools().writeContext();
        auto task = FbTaskCreator(writeContext.socialTxn(), sf::TaskState::Opened)
            .source("fbapi")
            .type(sf::Type::Barrier)
            .create();
        writeContext.commit();
    }

    // create feedback NOT within preset1 and other
    {
        auto writeContext = dbPools().writeContext();
        auto task = FbTaskCreator(writeContext.socialTxn(), sf::TaskState::Opened)
            .source("not-feedback-workflow")
            .type(sf::Type::Subway)
            .create();
        writeContext.commit();
    }

    // check feeds
    auto writeContext = dbPools().writeContext();
    sf::GatewayRO gatewayRo(writeContext.socialTxn());
    acl_utils::FeedbackPresetCheckerImpl fbPresetChecker(pool(), pool(), 10, std::chrono::seconds(10));

    // check feed for USER1
    {
        sf::TaskFilter filter;
        fbPresetChecker.restrictFilterByGlobalPresets(filter, USER1_ID);

        auto result = gatewayRo.tasksFeedWithCount(filter, FEED_PARAMS);

        UNIT_ASSERT_VALUES_EQUAL(result.tasks.size(), 3);
        UNIT_ASSERT_VALUES_EQUAL(result.tasks.at(0).id(), taskInPreset1And2);
        UNIT_ASSERT_VALUES_EQUAL(result.tasks.at(1).id(), taskInPreset1);
        UNIT_ASSERT_VALUES_EQUAL(result.tasks.at(2).id(), taskInPreset2);
    }

    // check feed for USER2
    {
        sf::TaskFilter filter;
        fbPresetChecker.restrictFilterByGlobalPresets(filter, USER2_ID);

        auto result = gatewayRo.tasksFeedWithCount(filter, FEED_PARAMS);

        UNIT_ASSERT_VALUES_EQUAL(result.tasks.size(), 0);
    }
}

}

} // namespace maps::wiki::socialsrv::tests
