#include "review_report.h"
#include "review_utils.h"

#include <yandex/maps/wiki/social/feedback/attribute_names.h>
#include <yandex/maps/wiki/social/feedback/attributes.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/retry_duration.h>

#include <maps/libs/auth/include/tvm.h>
#include <maps/libs/common/include/retry.h>
#include <maps/libs/common/include/temporary_dir.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/serialization.h>
#include <maps/libs/http/include/client.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/mds/mds.h>
#include <yandex/maps/shell_cmd.h>
#include <yandex/maps/shellcmd/logging_ostream.h>

#include <library/cpp/html/escape/escape.h>
#include <library/cpp/resource/resource.h>
#include <fmt/format.h>
#include <util/generic/guid.h>

#include <boost/algorithm/string/replace.hpp>
#include <fstream>
#include <map>
#include <regex>
#include <string>
#include <vector>

using namespace fmt::literals;
using namespace std::string_literals;

namespace maps::wiki::socialsrv {

namespace sf = social::feedback;

namespace {

const auto REPORT_HEADER_RESOURCE_ID = "report-header"s;
const auto REPORT_FOOTER_RESOURCE_ID = "report-footer"s;
const auto TOPIC_HEADER_RESOURCE_ID = "topic-header"s;
const auto RECORD_FULL_RESOURCE_ID = "record-full"s;
const auto RECORD_MAP_IMAGE_RESOURCE_ID = "record-map-image"s;
const auto RECORD_NO_IMAGE_RESOURCE_ID = "record-no-image"s;

const auto FEEDBACK_KEYSET_ID = "feedback"s;
const auto FEEDBACK_REVIEW_KEYSET_ID = "feedback-review"s;

const auto TRANSLATION_KEY_REGION_NAME = "report-region-name"s;
const auto TRANSLATION_KEY_PEDESTRIAN_LOGIN = "report-pedestrian-login"s;
const auto TRANSLATION_KEY_TASK_COMMENT = "report-task-comment"s;
const auto TRANSLATION_KEY_MAP = "report-map"s;
const auto TRANSLATION_KEY_IMAGE = "report-image"s;
const auto TRANSLATION_KEY_PEDESTRIAN_COMMENT = "report-pedestrian-comment"s;
const auto TRANSLATION_KEY_TASK_TYPE = "report-task-type"s;
const auto TRANSLATION_KEY_MAP_LINK = "report-map-link"s;

const std::map<sf::ReviewTaskComment::Topic, std::string> TOPIC_TO_TRANSLATION_KEY = {
    {sf::ReviewTaskComment::Topic::BadPhoto, "report-topic-bad-photo"},
    {sf::ReviewTaskComment::Topic::MissingObjects, "report-topic-missing-objects"},
    {sf::ReviewTaskComment::Topic::UnwantedObjects, "report-topic-unwanted-objects"},
    {sf::ReviewTaskComment::Topic::Other, "report-topic-other"}
};

const auto ATTR_NAME_CONTENT = "content"s;
const auto ATTR_NAME_IMAGE_FEATURES = "imageFeatures"s;
const auto ATTR_NAME_GEOMETRY = "geometry"s;
const auto ATTR_NAME_TARGET_GEOMETRY = "targetGeometry"s;
const auto ATTR_NAME_IMAGE_FULL = "imageFull"s;
const auto ATTR_NAME_URL = "url"s;

struct ImageInfo
{
    std::string url;
    geolib3::Point2 geometry;
    geolib3::Point2 targetGeometry;
};

struct TaskInfo
{
    std::string pedestrianComment;
    std::vector<ImageInfo> images;
};

ImageInfo
parseImageInfo(const json::Value& jsonImageFeature)
{
    REQUIRE(
        jsonImageFeature.hasField(ATTR_NAME_GEOMETRY) &&
        jsonImageFeature.hasField(ATTR_NAME_TARGET_GEOMETRY) &&
        jsonImageFeature.hasField(ATTR_NAME_IMAGE_FULL),
        "Invalid JSON for image " << jsonImageFeature);

    auto jsonImageFull = jsonImageFeature[ATTR_NAME_IMAGE_FULL];

    REQUIRE(
        jsonImageFull.hasField(ATTR_NAME_URL),
        "Invalid JSON for imageFull " << jsonImageFull);

    return ImageInfo {
        .url = jsonImageFull[ATTR_NAME_URL].as<std::string>(),
        .geometry = geolib3::readGeojson<geolib3::Point2>(
            jsonImageFeature[ATTR_NAME_GEOMETRY]),
        .targetGeometry = geolib3::readGeojson<geolib3::Point2>(
            jsonImageFeature[ATTR_NAME_TARGET_GEOMETRY])
    };
}

TaskInfo
parseTaskInfo(const sf::Task& task)
{
    std::vector<ImageInfo> images;
    if (task.attrs().exist(sf::AttrType::SourceContext)) {
        auto jsonSourceContext = task.attrs().get(sf::AttrType::SourceContext);

        REQUIRE(
            jsonSourceContext.hasField(ATTR_NAME_CONTENT),
            "Invalid task " << task.id());

        const auto& jsonContent = jsonSourceContext[ATTR_NAME_CONTENT];

        REQUIRE(
            jsonContent.hasField(ATTR_NAME_IMAGE_FEATURES),
            "Invalid task " << task.id());

        for (const auto& jsonImageFeature : jsonContent[ATTR_NAME_IMAGE_FEATURES]) {
            images.push_back(parseImageInfo(jsonImageFeature));
        }
    }

    auto pedestrianComment =
        task.attrs().existCustom(sf::attrs::USER_DATA_COMMENT)
        ? task.attrs().getCustom(sf::attrs::USER_DATA_COMMENT)
        : ""s;

    return TaskInfo{pedestrianComment, images};
}

std::string
prepareForHtml(const std::string& text)
{
    std::string result = NHtml::EscapeText(TString(text));
    boost::replace_all(result, "\n", "<br/>");
    return result;
}

// Example: https://static-maps.yandex.ru/1.x/?l=map&amp;pt=37.784826,44.703026,pmrdm~37.784896,44.703080&amp;z=19
std::string
getStaticMapsUrl(const geolib3::Point2& geometry, const geolib3::Point2& targetGeometry)
{
    return
        "https://static-maps.yandex.ru/1.x/?l=map&amp;pt=" +
        std::to_string(geometry.x()) + "," + std::to_string(geometry.y()) +
        ",pmrdm~" +
        std::to_string(targetGeometry.x()) + "," + std::to_string(targetGeometry.y()) +
        "&amp;z=19";
}

// Example: https://yandex.ru/maps?ll=37.784826%2C44.703026&z=20
std::string
getYandexMapsUrl(const geolib3::Point2& geometry)
{
    return
        "https://yandex.ru/maps?ll=" +
        std::to_string(geometry.x()) + "%2C" + std::to_string(geometry.y()) + "&z=20";
}

std::string
makeHeader(
    const std::string& reviewComment,
    const RegionData& region,
    TranslationProvider translationProvider)
{
    return fmt::format(
        (std::string)NResource::Find(REPORT_HEADER_RESOURCE_ID),
        "text_region_name"_a = translationProvider(
            FEEDBACK_REVIEW_KEYSET_ID, TRANSLATION_KEY_REGION_NAME),
        "text_pedestrian_login"_a = translationProvider(
            FEEDBACK_REVIEW_KEYSET_ID, TRANSLATION_KEY_PEDESTRIAN_LOGIN),
        "text_task_comment"_a = translationProvider(
            FEEDBACK_REVIEW_KEYSET_ID, TRANSLATION_KEY_TASK_COMMENT),
        "text_map"_a = translationProvider(
            FEEDBACK_REVIEW_KEYSET_ID, TRANSLATION_KEY_MAP),
        "text_image"_a = translationProvider(
            FEEDBACK_REVIEW_KEYSET_ID, TRANSLATION_KEY_IMAGE),
        "text_pedestrian_comment"_a = translationProvider(
            FEEDBACK_REVIEW_KEYSET_ID, TRANSLATION_KEY_PEDESTRIAN_COMMENT),
        "region_name"_a = prepareForHtml(region.title),
        "pedestrian_login"_a = prepareForHtml(region.pedestrianLogin),
        "review_comment"_a = prepareForHtml(reviewComment));
}

std::string
formatTask(
    const sf::Task& task,
    const std::string& taskComment,
    TranslationProvider translationProvider,
    ImageProvider imageProvider)
{
    const auto& taskInfo = parseTaskInfo(task);

    if (taskInfo.images.empty() && !taskInfo.pedestrianComment.empty()) {
        return fmt::format(
            (std::string)NResource::Find(RECORD_NO_IMAGE_RESOURCE_ID),
            "task_comment"_a = prepareForHtml(taskComment),
            "text_task_type"_a = translationProvider(
                FEEDBACK_REVIEW_KEYSET_ID, TRANSLATION_KEY_TASK_TYPE),
            "task_type"_a = translationProvider(
                FEEDBACK_KEYSET_ID, "task-type-"s.append(toString(task.type()))),
            "pedestrian_comment"_a = prepareForHtml(taskInfo.pedestrianComment));
    }

    std::string result;
    for (size_t i = 0; i < taskInfo.images.size(); ++i) {
        const auto& imageInfo = taskInfo.images[i];

        if (i == 0) {
            result += fmt::format(
                (std::string)NResource::Find(RECORD_FULL_RESOURCE_ID),
                "image_count"_a = taskInfo.images.size(),
                "task_comment"_a = prepareForHtml(taskComment),
                "static_maps_url"_a = getStaticMapsUrl(imageInfo.geometry, imageInfo.targetGeometry),
                "yandex_maps_url"_a = getYandexMapsUrl(imageInfo.geometry),
                "text_map_link"_a = translationProvider(
                    FEEDBACK_REVIEW_KEYSET_ID, TRANSLATION_KEY_MAP_LINK),
                "image_url"_a = imageProvider(imageInfo.url),
                "text_task_type"_a = translationProvider(
                    FEEDBACK_REVIEW_KEYSET_ID, TRANSLATION_KEY_TASK_TYPE),
                "task_type"_a = translationProvider(
                    FEEDBACK_KEYSET_ID, "task-type-"s.append(toString(task.type()))),
                "pedestrian_comment"_a = prepareForHtml(taskInfo.pedestrianComment));
        } else {
            result += fmt::format(
                (std::string)NResource::Find(RECORD_MAP_IMAGE_RESOURCE_ID),
                "static_maps_url"_a = getStaticMapsUrl(imageInfo.geometry, imageInfo.targetGeometry),
                "yandex_maps_url"_a = getYandexMapsUrl(imageInfo.geometry),
                "text_map_link"_a = translationProvider(
                    FEEDBACK_REVIEW_KEYSET_ID, TRANSLATION_KEY_MAP_LINK),
                "image_url"_a = imageProvider(imageInfo.url));
        }
    }
    return result;
}

std::string
makeBody(
    const std::map<TId, sf::ReviewTaskComment>& tasksComments,
    const serialize::TasksForReviewUI& tasks,
    TranslationProvider translationProvider,
    ImageProvider imageProvider)
{
    std::map<sf::ReviewTaskComment::Topic, std::map<TId, sf::ReviewTaskComment> > tasksCommentsByTopic;
    for (const auto& [taskId, taskComment] : tasksComments) {
        if (taskComment.topic) {
            tasksCommentsByTopic[*taskComment.topic][taskId] = taskComment;
        }
    }

    std::map<TId, size_t> taskIndexesById;
    for (size_t i = 0; i < tasks.size(); ++i) {
        taskIndexesById[tasks[i].id()] = i;
    }

    std::string result;
    for (const auto& [topic, tasksComments] : tasksCommentsByTopic) {
        result += fmt::format(
            (std::string)NResource::Find(TOPIC_HEADER_RESOURCE_ID),
            "topic"_a = translationProvider(
                FEEDBACK_REVIEW_KEYSET_ID, TOPIC_TO_TRANSLATION_KEY.at(topic)));

        for (const auto& [taskId, taskComment] : tasksComments) {
            result += formatTask(
                tasks[taskIndexesById.at(taskId)],
                taskComment.comment,
                translationProvider,
                imageProvider);
        }
    }
    return result;
}

std::string
makeFooter()
{
    return fmt::format((std::string)NResource::Find(REPORT_FOOTER_RESOURCE_ID));
}

} // namespace

std::string
makeHtmlReport(
    const sf::Review& review,
    const ReviewExternData& reviewExternData,
    TranslationProvider translationProvider,
    ImageProvider imageProvider)
{
    return
        makeHeader(
            review.comment(),
            reviewExternData.region,
            translationProvider) +
        makeBody(
            review.tasksComments(),
            reviewExternData.tasks,
            translationProvider,
            imageProvider) +
        makeFooter();
}

namespace {

const auto TVM_MRC_BROWSER_ALIAS = "mrc-browser";
const size_t RETRY_COUNT = 3;
const unsigned HTTP_OK = 200;

const auto INPUT_HTML_FILE_NAME = "report.html"s;
const auto OUTPUT_PDF_FILE_NAME = "report.pdf"s;

const std::regex IMAGE_ID_REGEX("/feature/(\\d+)/image");

std::string
uploadToMds(
    const std::string& filePath,
    const mds::Configuration& mdsConfig,
    TId reviewId)
{
    auto mdsPath = "review_reports/" +
        std::to_string(reviewId) + "_" + CreateGuidAsString() + ".pdf";

    return wiki::common::retryDuration([&] {
        mds::Mds mds(mdsConfig);
        std::ifstream file(filePath, std::ios::in | std::ios::binary);
        REQUIRE(file, "Failed to open file for reading: " << filePath);

        INFO() << "Upload to MDS started: '" << mdsPath << "'";
        auto response = mds.post(mdsPath, file);
        INFO() << "Upload to MDS completed";

        file.close();
        return mds.makeReadUrl(response.key());
    });
}

} // namespace

MrcImageInfo
getMrcImageInfo(
    const std::string& inputFeedbackUrl,
    const std::string& mrcBrowserUrl)
{
    INFO() << "Image URL from feedback: '" << inputFeedbackUrl << "'";

    std::smatch match;
    REQUIRE(
        std::regex_search(inputFeedbackUrl, match, IMAGE_ID_REGEX),
        "Invalid image URL " << inputFeedbackUrl);
    const auto imageId = match[1].str();
    INFO() << "Image MRC ID: '" << imageId << "'";

    INFO() << "MRC browser url: '" << mrcBrowserUrl << "'";
    return {
        .imageId = imageId,
        .downloadUrl = mrcBrowserUrl + "/internal/feature/" + imageId + "/image"
    };
}

std::string
downloadMrcImage(
    const std::string& imageUrl,
    const std::string& mrcBrowserUrl,
    const std::string& ticket,
    const maps::common::TemporaryDir& tempDir)
{
    const auto& mrcImageInfo = getMrcImageInfo(imageUrl, mrcBrowserUrl);

    INFO() << "Image download started: '" << mrcImageInfo.downloadUrl << "'";
    http::Client httpClient;
    const auto [responseBody, status] = httpClient.get(
        mrcImageInfo.downloadUrl,
        {{auth::SERVICE_TICKET_HEADER, ticket}},
        maps::common::RetryPolicy().setTryNumber(RETRY_COUNT));
    REQUIRE(status == HTTP_OK,
        "HTTP request failed: " << status << " " << mrcImageInfo.downloadUrl);
    INFO() << "Image download completed: " << mrcImageInfo.imageId;

    const auto imageFileName = mrcImageInfo.imageId + ".bin";
    std::ofstream imageFile((tempDir / imageFileName).string());
    imageFile << responseBody;
    imageFile.close();

    return imageFileName;
}

std::string
convertHtmlToPdf(
    const std::string& htmlContent,
    const maps::common::TemporaryDir& tempDir)
{
    auto inputFilePath = (tempDir / INPUT_HTML_FILE_NAME).string();
    auto outputFilePath = (tempDir / OUTPUT_PDF_FILE_NAME).string();

    std::ofstream inputFile(inputFilePath);
    inputFile << htmlContent;
    inputFile.close();

    shell::stream::LoggingOutputStream loggedOut(
        [](const std::string& s){ INFO() << s; });
    shell::stream::LoggingOutputStream loggedErr(
        [](const std::string& s){ ERROR() << s; });

    INFO() << "HTML --> PDF conversion started";
    auto convertCmd = "wkhtmltopdf " + inputFilePath + " " + outputFilePath;
    shell::ShellCmd cmd(convertCmd, loggedOut, loggedErr);
    auto exitCode = cmd.run();
    REQUIRE(!exitCode, "wkhtmltopdf failed: " << convertCmd);
    INFO() << "HTML --> PDF conversion completed";

    return outputFilePath;
}

std::string
makePdfReportAndUpload(
    const social::feedback::Review& review,
    const ReviewExternData& reviewExternData,
    const std::string& lang,
    const json::Value& translations,
    const NTvmAuth::TTvmClient& tvmClient,
    const mds::Configuration& mdsConfig,
    const std::string& mrcBrowserUrl)
{
    INFO() << "Report language: '" << lang << "'";

    auto ticket = tvmClient.GetServiceTicketFor(TVM_MRC_BROWSER_ALIAS);
    maps::common::TemporaryDir tempDir;

    const auto htmlContent = makeHtmlReport(
        review,
        reviewExternData,
        [&](const std::string& keyset, const std::string& key) {
            if (!translations.hasField(lang)) {
                return ""s;
            }
            return translations[lang][keyset][key].as<std::string>();
        },
        [&](const std::string& imageUrl) {
            return downloadMrcImage(imageUrl, mrcBrowserUrl, ticket, tempDir);
        });

    return uploadToMds(
        convertHtmlToPdf(htmlContent, tempDir),
        mdsConfig,
        review.id());
}

} // namespace maps::wiki::socialsrv
