#include <maps/wikimap/mapspro/services/social/src/libs/feedback/review_report.h>
#include <maps/wikimap/mapspro/libs/social/tests/helpers/fb_review_factory.h>
#include <maps/wikimap/mapspro/libs/social/tests/helpers/fb_task_factory.h>

#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/common/include/temporary_dir.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/json/include/value.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <filesystem>

using namespace std::string_literals;

namespace maps::wiki::socialsrv::tests {

namespace sf = social::feedback;

namespace {

const TId TASK_ID_1 = 1;
const TId TASK_ID_2 = 2;
const TId TASK_ID_3 = 3;
const TId TASK_ID_4 = 4;

const TUid CREATED_BY = 1;
const TUid REVIEWEE = 2;
const TId REGION_ID = 3;
const TId REGION_COMMIT_ID = 4;

const auto REGION_CATEGORY = "mrc_pedestrian_region"s;
const auto SOURCE = "partner-pedestrian-onfoot"s;

const geolib3::Polygon2 REGION_GEOMETRY {
    geolib3::PointsVector{
        {0, 0},
        {0, 1},
        {1, 1},
        {1, 0},
        {0, 0}
    }
};

const auto TRANSLATIONS_RESOURCE_ID = "translations"s;

const auto TEST_MRC_BROWSER_URL = "http://core-nmaps-mrc-browser-pro.common.testing.maps.yandex.ru"s;

struct TestReviewData
{
    const social::feedback::Review& review;
    const ReviewExternData& reviewExternData;
};

// ya tool tvmknife get_service_ticket sshkey -s 2027488 -d 2011488
const auto DEFAULT_TVM_TICKET = "<ticket>"s;
std::string getTestTvmTicket()
{
    return DEFAULT_TVM_TICKET;
}

} // namespace

Y_UNIT_TEST_SUITE(feedback_review_report) {

Y_UNIT_TEST(mrc_image_info)
{
    const auto mrcImageInfo = getMrcImageInfo(
        "https://core-nmaps-mrc-browser-datatesting.crowdtest.maps.yandex.ru/feature/187105075/image",
        "http://core-nmaps-mrc-browser-pro.common.datatesting.maps.yandex.ru");
    UNIT_ASSERT_STRINGS_EQUAL(mrcImageInfo.imageId, "187105075");
    UNIT_ASSERT_STRINGS_EQUAL(
        mrcImageInfo.downloadUrl,
        "http://core-nmaps-mrc-browser-pro.common.datatesting.maps.yandex.ru/internal/feature/187105075/image");
}

Y_UNIT_TEST(feedback_review_report)
{
    auto taskFactory = sf::tests::FbTaskFactory().type(sf::Type::PedestrianRoute).source(SOURCE);
    auto task1 = taskFactory.id(TASK_ID_1).attrs(
        json::Value::fromString(common::readFileToString(SRC_("data/task_attrs_1.json")))).create();
    auto task2 = taskFactory.id(TASK_ID_2).attrs(
        json::Value::fromString(common::readFileToString(SRC_("data/task_attrs_2.json")))).create();
    auto task3 = taskFactory.id(TASK_ID_3).attrs(
        json::Value::fromString(common::readFileToString(SRC_("data/task_attrs_3.json")))).create();
    auto task4 = taskFactory.id(TASK_ID_4).attrs(
        json::Value::fromString(common::readFileToString(SRC_("data/task_attrs_4.json")))).create();

    auto review = sf::tests::FbReviewFactory::makeReview(
        CREATED_BY, REVIEWEE, REGION_ID, REGION_COMMIT_ID);
    review.setComment("В целом удовлетворительно.\nЕсть отдельные замечания.");
    review.setTaskComment(task1.id(),
        "Нет сигнала на афишу <> забор", sf::ReviewTaskComment::Topic::MissingObjects);
    review.setTaskComment(task2.id(),
        "Несколько одинаковых фотографий", sf::ReviewTaskComment::Topic::BadPhoto);
    review.setTaskComment(task3.id(),
        "Непонятна конфигурация дорожек", sf::ReviewTaskComment::Topic::BadPhoto);
    review.setTaskComment(task4.id(),
        "Нет фото", sf::ReviewTaskComment::Topic::BadPhoto);

    ReviewExternData reviewExternData;
    reviewExternData.region = {
        .id = REGION_ID,
        .commitId = REGION_COMMIT_ID,
        .categoryId = REGION_CATEGORY,
        .title = "Зона " + std::to_string(REGION_ID),
        .pedestrianUid = CREATED_BY,
        .pedestrianLogin = "walker" + std::to_string(CREATED_BY),
        .polygon = REGION_GEOMETRY
    };
    reviewExternData.tasks.push_back(serialize::TaskForReviewUI(task1));
    reviewExternData.tasks.push_back(serialize::TaskForReviewUI(task2));
    reviewExternData.tasks.push_back(serialize::TaskForReviewUI(task3));
    reviewExternData.tasks.push_back(serialize::TaskForReviewUI(task4));

    const auto& translations = json::Value::fromResource(TRANSLATIONS_RESOURCE_ID);

    if (getTestTvmTicket() == DEFAULT_TVM_TICKET) {
        auto htmlContent = makeHtmlReport(
            review,
            reviewExternData,
            [&](const std::string& keyset, const std::string& key) {
                return translations["ru"][keyset][key].as<std::string>();
            },
            [&](const std::string& imageUrl) {
                return imageUrl;
            });

        UNIT_ASSERT_STRINGS_EQUAL(
            htmlContent,
            common::readFileToString(SRC_("data/review_report.html")));
    } else {
        maps::common::TemporaryDir tempDir;
        auto htmlContent = makeHtmlReport(
            review,
            reviewExternData,
            [&](const std::string& keyset, const std::string& key) {
                return translations["ru"][keyset][key].as<std::string>();
            },
            [&](const std::string& imageUrl) {
                return downloadMrcImage(
                    imageUrl,
                    TEST_MRC_BROWSER_URL,
                    getTestTvmTicket(),
                    tempDir);
            });

        UNIT_ASSERT_NO_EXCEPTION(convertHtmlToPdf(htmlContent, tempDir));
        //std::filesystem::copy(tempDir / "report.pdf", "/home/<login>/report.pdf");
    }
}

}

} // namespace maps::wiki::socialsrv::tests
