#include "helpers.h"
#include <maps/wikimap/mapspro/libs/acl/include/common.h>
#include <maps/libs/geolib/include/conversion.h>

namespace maps::wiki::socialsrv {

void checkUid(acl::UID uid)
{
    SOCIAL_REQUIRE(uid > 0, Forbidden, "Invalid uid: " << uid);
}

std::optional<chrono::TimePoint> parseOptionalTimePoint(
    const yacare::Request& request,
    std::string_view name)
{
    const auto& valueStr = request.input()[name];
    if (valueStr.empty()) {
        return std::nullopt;
    }
    try {
        return chrono::parseIsoDateTime(valueStr);
    }
    catch (const std::exception&) {
        throw yacare::errors::BadRequest() << "invalid " << name;
    }
}

std::optional<social::DateTimeCondition> parseOptionalDateTimeCondition(
    const yacare::Request& request,
    std::string_view nameBefore, std::string_view nameAfter)
{
    auto timePointBefore = parseOptionalTimePoint(request, nameBefore);
    auto timePointAfter = parseOptionalTimePoint(request, nameAfter);
    if (timePointBefore || timePointAfter) {
        return social::DateTimeCondition(timePointAfter, timePointBefore);
    }
    return std::nullopt;
}

void makeJsonResponse(
    yacare::Response& response,
    const std::string& jsonString)
{
    response << jsonString;
    response.setHeader("Content-Type", "application/json; charset=UTF-8");
}

std::string jsonToStringNoThrow(const json::Value& json)
{
    if (json.exists()) {
        return (json::Builder() << json).str();
    }
    return "<Nonexistent json value>";
}

geolib3::Point2 pointFromGeojson(const json::Value& geojson)
{
    try {
        return geolib3::readGeojson<geolib3::Point2>(geojson);
    } catch (const std::exception& e) {
        throw yacare::errors::BadRequest()
            << "Error parsing Point2 from geojson: '"
            << jsonToStringNoThrow(geojson) << "': " << e.what();
    }
}

std::optional<social::DateTimeCondition> parseOptionalDateTimeCondition(
    const json::Value& jsonBefore, const json::Value& jsonAfter)
{
    auto timePointBefore = parseOptional<chrono::TimePoint>(jsonBefore);
    auto timePointAfter = parseOptional<chrono::TimePoint>(jsonAfter);
    if (timePointBefore || timePointAfter) {
        return social::DateTimeCondition(timePointAfter, timePointBefore);
    }
    return std::nullopt;
}

social::feedback::TaskFeedParamsId getFeedParams(const yacare::Request& request)
{
    auto beforeId = queryParam<social::TId>(request, "before", 0);
    auto afterId = queryParam<social::TId>(request, "after", 0);
    auto perPage = queryParam<size_t>(request, "per-page", 10);

    REQUIRE(
        beforeId == 0 || afterId == 0,
        yacare::errors::BadRequest()
            << "Both non-zero before and after are not allowed"
    );

    return social::feedback::TaskFeedParamsId(
        beforeId,
        afterId,
        perPage,
        social::TasksOrder::OldestFirst
    );
}

} //namespace maps::wiki::socialsrv
