#include <maps/wikimap/mapspro/services/social_backoffice/src/api/globals.h>

#include <maps/wikimap/mapspro/libs/acl_utils/include/feedback.h>
#include <maps/wikimap/mapspro/services/social/src/libs/feedback/serialize.h>
#include <maps/wikimap/mapspro/services/social/src/libs/yacare/helpers.h>

#include <maps/wikimap/mapspro/services/social/src/libs/feedback-actions/tasks_feed.h>
#include <maps/wikimap/mapspro/services/social/src/libs/feedback-actions/tasks_from_json.h>
#include <maps/wikimap/mapspro/services/social/src/libs/feedback-actions/tasks_get.h>
#include <maps/wikimap/mapspro/services/social/src/libs/feedback-actions/tasks_helpers.h>
#include <maps/wikimap/mapspro/services/social/src/libs/feedback-actions/tasks_poi.h>
#include <maps/wikimap/mapspro/services/social/src/libs/feedback-actions/tasks_resolve.h>

#include <yandex/maps/wiki/common/robot.h>


namespace maps::wiki::socialsrv {

namespace mwsf = social::feedback;

namespace {

const auto ROBOT_UID = common::ROBOT_UID;
const auto EMPTY_TOKEN = "";

yacare::ThreadPool feedbackTasksThreadPool("feedbackTasksThreadPool", 4, 2048);

class FeedbackPresetCheckerTrusted : public acl_utils::FeedbackPresetChecker {
public:
    FeedbackPresetCheckerTrusted() = default;

    bool isInAssignedPreset(acl::UID, const sf::Task&, acl_utils::PresetRightKind) const override { return true; }
    acl_utils::AssignedPresetsExt getAssignedPresets(social::TUid) const override {
        return acl_utils::AssignedPresetsExt(std::chrono::steady_clock::now());
    }
    void restrictFilterByGlobalPresets(sf::TaskFilter&, social::TUid) const override {}
    void restrictFilterByGlobalPresets(sf::MvSourceTypeFilter&, social::TUid) const override {}
    bool hasModifyPermission(acl::UID, const sf::Preset&, social::TId) const override {
        return true;
    }
};


} // namespace

YCR_USE(feedbackTasksThreadPool) {

YCR_RESPOND_TO("GET /feedback/tasks")
{
    TasksFeedParams params;
    params.afterId = queryParam<ID>(request, "after", 0);
    params.perPage = queryParam<size_t>(request, "per-page", params.perPage);
    params.types = optionalVectorQueryParam<mwsf::Type>(request, "type");
    params.bboxGeo = optionalQueryParam<std::string>(request, "bbox-geo");
    params.sources = optionalVectorQueryParam<std::string>(request, "source");
    params.workflows = optionalVectorQueryParam<mwsf::Workflow>(request, "workflow");

    auto feedForUI = getTasksFeed(
        Globals::dbPools(),
        acl_utils::FeedbackChecker(
            Globals::aclChecker(),
            FeedbackPresetCheckerTrusted()
        ),
        params,
        ROBOT_UID,
        EMPTY_TOKEN);

    makeJsonResponse(response, toJson(feedForUI, ROBOT_UID));
}

YCR_RESPOND_TO("GET /feedback/tasks/$")
{
    const auto taskId = positionalParam<social::TId>(argv, 0);

    auto taskExtended = getTask(
        Globals::dbPools(),
        acl_utils::FeedbackChecker(
            Globals::aclChecker(),
            FeedbackPresetCheckerTrusted()
        ),
        taskId,
        ROBOT_UID,
        EMPTY_TOKEN);

    makeJsonResponse(response, toJson(taskExtended, ROBOT_UID));
}

YCR_RESPOND_TO("POST /feedback/tasks")
{
    auto newTask = createNewTaskFromJson(json::Value::fromString(request.body()));

    auto result = postNewTask(Globals::dbPools(), newTask);
    makeJsonResponse(response, toJson(result.taskExt, result.token, ROBOT_UID));
}

YCR_RESPOND_TO("POST /feedback/tasks/poi")
{
    auto requestBody = maps::json::Value::fromString(request.body());
    auto newTask = poiNewTaskFromJson(requestBody);

    auto result = createPoiTask(Globals::dbPools(), newTask);
    makeJsonResponse(response, toJson(result.taskExt, result.token, ROBOT_UID));
}

YCR_RESPOND_TO("POST /feedback/tasks/$/resolve")
{
    const auto taskId = positionalParam<social::TId>(argv, 0);
    const auto verdict = queryParam<mwsf::Verdict>(request, "resolution");
    const auto rejectReason = mwsf::RejectReason::NoData;

    auto result = resolveTask(
        Globals::dbPools(),
        acl_utils::FeedbackChecker(
            Globals::aclChecker(),
            FeedbackPresetCheckerTrusted()
        ),
        taskId, ROBOT_UID, verdict, rejectReason, request.body());

    makeJsonResponse(response, toJson(result.taskExt, result.token, ROBOT_UID));
}

YCR_RESPOND_TO("POST /feedback/tasks/road-direction")
{
    auto requestBody = maps::json::Value::fromString(request.body());
    auto newTask = roadDirectionNewTaskFromJson(requestBody);

    auto result = postNewTask(Globals::dbPools(), newTask);
    makeJsonResponse(response, toJson(result.taskExt, result.token, ROBOT_UID));
}

YCR_RESPOND_TO("POST /feedback/tasks/speed-limit")
{
    auto newTask = speedLimitNewTaskFromJson(maps::json::Value::fromString(request.body()));

    auto result = postNewTask(Globals::dbPools(), newTask);
    makeJsonResponse(response, toJson(result.taskExt, result.token, ROBOT_UID));
}

YCR_RESPOND_TO("POST /feedback/tasks/absent-barrier")
{
    auto newTask = absentBarrierNewTaskFromJson(maps::json::Value::fromString(request.body()));

    auto result = postNewTask(Globals::dbPools(), newTask);
    makeJsonResponse(response, toJson(result.taskExt, result.token, ROBOT_UID));
}

YCR_RESPOND_TO("POST /feedback/tasks/absent-entrance")
{
    auto newTask = absentEntranceNewTaskFromJson(maps::json::Value::fromString(request.body()));

    auto result = postNewTask(Globals::dbPools(), newTask);
    makeJsonResponse(response, toJson(result.taskExt, result.token, ROBOT_UID));
}

YCR_RESPOND_TO("POST /feedback/tasks/parking")
{
    auto newTask = parkingNewTaskFromJson(maps::json::Value::fromString(request.body()));

    auto result = postNewTask(Globals::dbPools(), newTask);
    makeJsonResponse(response, toJson(result.taskExt, result.token, ROBOT_UID));
}

YCR_RESPOND_TO("POST /feedback/tasks/parking-ft-type")
{
    auto newTask = parkingFtTypeNewTaskFromJson(maps::json::Value::fromString(request.body()));

    auto result = postNewTask(Globals::dbPools(), newTask);
    makeJsonResponse(response, toJson(result.taskExt, result.token, ROBOT_UID));
}

YCR_RESPOND_TO("POST /feedback/tasks/traffic-sign")
{
    auto newTask = trafficSignNewTaskFromJson(maps::json::Value::fromString(request.body()));

    auto result = postNewTask(Globals::dbPools(), newTask);
    makeJsonResponse(response, toJson(result.taskExt, result.token, ROBOT_UID));
}

YCR_RESPOND_TO("POST /feedback/tasks/traffic-sign-disappearance")
{
    auto newTask = trafficSignDisappearanceNewTaskFromJson(maps::json::Value::fromString(request.body()));

    auto result = postNewTask(Globals::dbPools(), newTask);
    makeJsonResponse(response, toJson(result.taskExt, result.token, ROBOT_UID));
}

YCR_RESPOND_TO("POST /feedback/tasks/prohibited-path")
{
    auto newTask = prohibitedPathNewTaskFromJson(
        maps::json::Value::fromString(request.body())
    );

    auto result = postNewTask(Globals::dbPools(), newTask);
    makeJsonResponse(response, toJson(result.taskExt, result.token, ROBOT_UID));
}

YCR_RESPOND_TO("POST /feedback/tasks/absent-address")
{
    auto newTask = absentAddressNewTaskFromJson(maps::json::Value::fromString(request.body()));

    auto result = postNewTask(Globals::dbPools(), newTask);
    makeJsonResponse(response, toJson(result.taskExt, result.token, ROBOT_UID));
}

YCR_RESPOND_TO("POST /feedback/tasks/onfoot", uid)
{
    auto newTask = onFootNewTaskFromJson(
        Globals::dbPools(),
        uid,
        maps::json::Value::fromString(request.body()));

    auto result = postNewTask(Globals::dbPools(), newTask);
    makeJsonResponse(response, toJson(result.taskExt, result.token, uid));
}

YCR_RESPOND_TO("POST /feedback/tasks/route-gap")
{
    auto newTask = rootGapNewTaskFromJson(maps::json::Value::fromString(request.body()));

    auto result = postNewTask(Globals::dbPools(), newTask);
    makeJsonResponse(response, toJson(result.taskExt, result.token, ROBOT_UID));
}

YCR_RESPOND_TO("POST /feedback/tasks/absent-traffic-light")
{
    auto newTask = absentTrafficLightNewTaskFromJson(maps::json::Value::fromString(request.body()));

    auto result = postNewTask(Globals::dbPools(), newTask);
    makeJsonResponse(response, toJson(result.taskExt, result.token, ROBOT_UID));
}

YCR_RESPOND_TO("POST /feedback/tasks/speed-bump")
{
    auto newTask = speedBumpSignNewTaskFromJson(maps::json::Value::fromString(request.body()));

    auto result = postNewTask(Globals::dbPools(), newTask);
    makeJsonResponse(response, toJson(result.taskExt, result.token, ROBOT_UID));
}

} // YCR_USE

} // namespace maps::wiki::socialsrv
