#include "globals.h"

#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <maps/libs/auth/include/tvm.h>
#include <maps/libs/common/include/environment.h>

#include <memory>
#include <set>

namespace maps::wiki::socialsrv {

namespace {

const std::string CORE_DB_ID = "core";
const std::string SOCIAL_DB_ID = "social";
const std::string VIEW_TRUNK_DB_ID = "view-trunk";
const std::string POOL_ID = "social";

const size_t ACL_CACHE_MAX_SIZE = 10 * 1000;
constexpr std::chrono::seconds ACL_CACHE_EXPIRATION_PERIOD(300);

const std::set<maps::common::Environment> SUPPORTED_ENVIRONMENTS {
    maps::common::Environment::Unstable,
    maps::common::Environment::Testing,
    maps::common::Environment::Stable,
};

struct GlobalData
{
    explicit GlobalData(const common::ExtendedXmlDoc& config);

    common::PoolHolder corePoolHolder;
    common::PoolHolder socialPoolHolder;
    common::PoolHolder viewTrunkPoolHolder;
    DbPoolsWithViewTrunk dbpools;

    acl_utils::CachingAclChecker aclChecker;

    std::shared_ptr<NTvmAuth::TTvmClient> tvmClient;
    std::unique_ptr<geosearch_client::Client> geosearchClient;
};

std::shared_ptr<NTvmAuth::TTvmClient> createTvmClient()
{
    if (!SUPPORTED_ENVIRONMENTS.contains(maps::common::getYandexEnvironment())) {
        return {};
    }
    return std::make_shared<NTvmAuth::TTvmClient>(
        maps::auth::TvmtoolSettings().makeTvmClient());
}


GlobalData::GlobalData(const common::ExtendedXmlDoc& config)
    : corePoolHolder(config, CORE_DB_ID, POOL_ID)
    , socialPoolHolder(config, SOCIAL_DB_ID, POOL_ID)
    , viewTrunkPoolHolder(config, VIEW_TRUNK_DB_ID, POOL_ID)
    , dbpools(corePoolHolder.pool(),
        socialPoolHolder.pool(),
        viewTrunkPoolHolder.pool())

    , aclChecker(corePoolHolder.pool(),
        ACL_CACHE_MAX_SIZE,
        ACL_CACHE_EXPIRATION_PERIOD)

    , tvmClient(createTvmClient())
{
    geosearch_client::Configuration configuration(
        config.get<std::string>("/config/common/geocoder/base-url"),
        "maps-core-nmaps-social-backoffice");

    configuration.setTimeout(std::chrono::seconds(3));
    if (tvmClient) {
        configuration.setTvmTicketProvider(
            [tvmClient = this->tvmClient]() {
                return tvmClient->GetServiceTicketFor("geosearch");
            });
    }
    geosearchClient = std::make_unique<geosearch_client::Client>(
        std::move(configuration)
    );
}

std::unique_ptr<GlobalData> g_globalData;

} // namespace

GlobalsScope::GlobalsScope(const common::ExtendedXmlDoc& config)
{
    g_globalData = std::make_unique<GlobalData>(config);
}

GlobalsScope::~GlobalsScope()
{
    g_globalData.reset();
}

DbPoolsWithViewTrunk& Globals::dbPools()
{
    ASSERT(g_globalData);
    return g_globalData->dbpools;
}

acl_utils::CachingAclChecker& Globals::aclChecker()
{
    ASSERT(g_globalData);
    return g_globalData->aclChecker;
}

geosearch_client::Client& Globals::geosearchClient()
{
    ASSERT(g_globalData);
    ASSERT(g_globalData->geosearchClient);
    return *g_globalData->geosearchClient;
}

} // namespace maps::wiki::socialsrv
