#include <maps/wikimap/mapspro/services/social_backoffice/src/api/globals.h>
#include <maps/wikimap/mapspro/services/social_backoffice/src/lib/poi.h>
#include <maps/wikimap/mapspro/services/social_backoffice/src/lib/sprav.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/enum_io/include/enum_io.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/default_config.h>
#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/string_utils.h>
#include <memory>

namespace maps::wiki::socialsrv {
namespace {

enum class Action
{
    SpravInfo,
};

constexpr enum_io::Representations<Action> ACTION_REPR {
    {Action::SpravInfo, "sprav-info"},
};
DEFINE_ENUM_IO(Action, ACTION_REPR);

std::string allActions()
{
    return common::join(maps::enum_io::enumerateValues<Action>(), '\n');
}

int actionSpravInfo(social::TId permalinkId, double distance)
{
    auto spravData = loadSpravData(Globals::geosearchClient(), permalinkId);
    if (!spravData) {
        ERROR() << "Can not load sprav data for permalink : " << permalinkId;
        return EXIT_FAILURE;
    }
    INFO() << "Permalinks: " << common::join(spravData->clusterPermalinkIds, ',');
    INFO() << "Object ids: " << common::join(spravData->objectIds, ',');

    const auto& pos = spravData->positionGeo;
    INFO() << "Position x: " << std::to_string(pos.x()) << " y: " << std::to_string(pos.y());

    auto companyData = findCompany(Globals::dbPools(), *spravData, distance);
    if (!companyData) {
        ERROR() << "Can not load commany by sprav data";
        return EXIT_FAILURE;
    }
    INFO() << "Loaded POI: " << companyData->business.objectId;
    return EXIT_SUCCESS;
}

int run(
    const common::ExtendedXmlDoc& config,
    const std::string& actionStr,
    social::TId permalinkId,
    double distance)
{
    INFO() << "Action: " << actionStr;

    auto action = enum_io::tryFromString<Action>(actionStr);
    if (!action) {
        ERROR() << "Unsupported action: " << actionStr;
        return EXIT_FAILURE;
    }

    maps::wiki::socialsrv::GlobalsScope globalsScope(config);

    switch(*action) {
        case Action::SpravInfo:
            return actionSpravInfo(permalinkId, distance);
    }
    return EXIT_SUCCESS;
}

} // namespace
} // namespace maps::wiki::socialsrv

int main(int argc, char** argv) try {
    using namespace maps::wiki::socialsrv;

    maps::cmdline::Parser argsParser;
    auto config = argsParser.file("config")
        .help("Path to services.xml with database connection settings.");
    auto action = argsParser.string("action")
        .help(allActions());
    auto permalinkId = argsParser.size_t("permalink-id")
        .defaultValue(0)
        .help("permalink id from Sprav");
    auto distance = argsParser.real("distance")
        .defaultValue(0)
        .help("distance to poi");
    argsParser.parse(argc, argv);

    if (!action.defined()) {
        argsParser.printUsageAndExit(EXIT_FAILURE);
    }

    auto configPtr = config.defined()
        ? std::make_unique<maps::wiki::common::ExtendedXmlDoc>(config)
        : maps::wiki::common::loadDefaultConfig();

    return run(*configPtr, action, permalinkId, distance);
} catch (const maps::Exception& e) {
    FATAL() << "Exception caught: " << e;
    return EXIT_FAILURE;
} catch (const std::exception& ex) {
    FATAL() << "Exception caught: " << ex.what();
    return EXIT_FAILURE;
} catch (...) {
    FATAL() << "Unknown exception caught";
    return EXIT_FAILURE;
}
