#include <maps/wikimap/mapspro/services/tasks-ng/modules/acl.h>
#include <maps/wikimap/mapspro/services/tasks-ng/modules/common.h>

#include <yandex/maps/wiki/common/string_utils.h>

namespace maps {
namespace wiki {
namespace tasks_ng {
namespace modules {
namespace badge {

const std::string TASK_TYPE = "badge";
const std::string TABLE_TASK = sql::SCHEMA + "." + TASK_TYPE + "_task";
const std::string FIELDS_TASK = "action, badge_id, result_url";

const acl::SubjectPath ACL_PATH("mpro/tasks/badge");

class Context : public TaskContext
{
public:
    explicit Context(const pqxx::row& row)
        : action(row[0].as<std::string>())
        , badgeId(row[1].as<std::string>())
        , resultUrl(row[2].as<std::string>({}))
    {}

    void write(XmlWriter& writer, TaskWriteMode) const override
    {
        writer.addTag("badge-context", [&] {
            writer
                .addTagData("action", action)
                .addTagData("badge-id", badgeId);
        });
    }

    const std::string action;
    const std::string badgeId;
    const std::string resultUrl;
};

class Result : public TaskResult
{
public:
    explicit Result(const std::string& url_)
        : url(url_)
    {}

    void write(XmlWriter& writer, TaskWriteMode) const override
    {
        writer.addTag("badge-result", [&] {
            writer.addTagData("url", url);
        });
    }

    const std::string url;
};

class Module : public TaskModule
{
public:
    std::string name() const override { return TASK_TYPE; }

    void loadContext(DbContext& ctx, Task& task) const override
    {
        task.loadContext<Context>(ctx, TABLE_TASK, FIELDS_TASK);
    }

    void loadResult(DbContext&, Task& task) const override
    {
        auto contextPtr = dynamic_cast<Context*>(task.data().context.get());
        ASSERT(contextPtr);
        if (!contextPtr->resultUrl.empty()) {
            task.loadResult<Result>(contextPtr->resultUrl);
        }
    }

    void onCreate(
        DbContext& ctx,
        Task& task,
        const RequestParameters& parameters) const override
    {
        auto file = parameters.getFile("data");
        TASKS_REQUIRE(file.has_value(), ERR_BADGE_FILE_NOT_POSTED, "No file in the request");

        //Example: badge_id.action.txt
        auto parts = common::split(file->fileName, ".");
        TASKS_REQUIRE(parts.size() == 3, ERR_BAD_REQUEST, "Wrong filename format '" << file->fileName << "'");
        TASKS_REQUIRE(parts[2] == ".txt", ERR_BAD_REQUEST, "Wrong filename extension");

        const auto& badgeId = parts[0];
        TASKS_REQUIRE(!badgeId.empty(), ERR_BAD_REQUEST, "Empty badge_id");

        const auto& action = parts[1];
        TASKS_REQUIRE(action == "add" || action == "remove", ERR_BAD_REQUEST, "Unknown action '" << action << "'");

        checkPermission(ctx, task, ACL_PATH);

        auto& txnCore = ctx.txnCore();

        auto query = Query()
            .insertInto(TABLE_TASK)
            .columns("id, " + FIELDS_TASK + ",file")
            .values(Query()
                << task.id() << ','
                << txnCore.quote(action) << ','
                << txnCore.quote(badgeId) << ','
                << "NULL, " //result_url
                << "'" << txnCore.esc_raw(file->fileBody) << "'")
            .returning(FIELDS_TASK);

        auto rows = txnCore.exec(query.str());
        ASSERT(rows.size() == 1);

        task.setContext(std::make_unique<Context>(rows[0]));
    }

    json::Value launchParameters(const Task& task) const override
    {
        const auto& ctx = task.context<Context>();

        return json::Value{
            { TYPE, json::Value{TASK_TYPE} },
            { TASK_ID, json::Value{task.id()} },
            { "action", json::Value{ctx.action} },
            { "badgeId", json::Value{ctx.badgeId} }
        };
    }
};

const auto module = TaskModuleRegistry::get().registerModule<Module>();

} // namespace badge
} // namespace modules
} // namespace tasks_ng
} // namespace wiki
} // namespace maps
