#include <maps/wikimap/mapspro/services/tasks-ng/modules/common.h>
#include <maps/wikimap/mapspro/services/tasks-ng/modules/revision_helpers.h>

namespace maps {
namespace wiki {
namespace tasks_ng {
namespace modules {
namespace export_ {

const std::string TASK_TYPE = "export";

const std::string TABLE_TASK = sql::SCHEMA + "." + TASK_TYPE + "_task";
const std::string TABLE_RESULT = sql::SCHEMA + "." + TASK_TYPE + "_result";

const std::string FIELDS_TASK = "commit_id, subset, branch_id, tested";
const std::string FIELDS_RESULT = "message";

const std::vector<std::string> SUBSETS {
    "service",
    "domain",
    "masstransit",
};

class Context : public TaskContext
{
public:
    explicit Context(const pqxx::row& row)
        : commitId(row[0].as<Id>(0))
        , subset(row[1].as<std::string>({}))
        , branchId(row[2].as<Id>())
        , tested(row[3].as<bool>())
    {}

    void write(XmlWriter& writer, TaskWriteMode) const override
    {
        writer.addTag("export-context", [&] {
            writer
                .addTagData("branch", branchId)
                .addTagData("commit-id", commitId)
                .addTagData("subset", subset)
                .addTagData("tested", tested);
        });
    }

    bool revokable() const override { return true; }


    const Id commitId;
    const std::string subset;
    const Id branchId;
    const bool tested;
};

class Result : public TaskResult
{
public:
    Result(DbContext& ctx, Id taskId)
    {
        auto query = Query()
            .select(FIELDS_RESULT)
            .from(TABLE_RESULT)
            .where() << "task_id=" << taskId;
        auto rows = ctx.txnCore().exec(query.str());

        for (const auto& row : rows) {
            auto json = json::Value::fromString(row[0].as<std::string>());
            ASSERT(json.isObject());

            for (const auto& key : json.fields()) {
                urls_[key] = json[key].toString();
            }
        }
    }

    void write(XmlWriter& writer, TaskWriteMode mode) const override
    {
        if (mode == TaskWriteMode::Brief) {
            return;
        }

        writer.addTag("export-result", [&] {
            for (const auto& pair : urls_) {
                writer.addTag("uri", [&] {
                    writer.addAttribute("description", pair.first);
                    writer << pair.second;
                });
            }
        });
    }

private:
    std::map<std::string, std::string> urls_;
};

class Module : public TaskModule
{
public:
    std::string name() const override { return TASK_TYPE; }

    void capabilities(XmlWriter& writer) const override
    {
        writer.addTag("export-task-type", [&] {
            writer.addTag("subsets", [&] {
                for (const auto& subset : SUBSETS) {
                    writer.addTagData("subset", subset);
                }
            });
        });
    }

    void loadContext(DbContext& ctx, Task& task) const override
    {
        task.loadContext<Context>(ctx, TABLE_TASK, FIELDS_TASK);
    }

    void loadResult(DbContext& ctx, Task& task) const override
    {
        task.loadResult<Result>(ctx, task.id());
    }

    void onCreate(
        DbContext& ctx,
        Task& task,
        const RequestParameters& parameters) const override
    {
        auto subset = parameters.getValue<std::string>("subset", "");
        TASKS_REQUIRE(
            std::find(SUBSETS.begin(), SUBSETS.end(), subset) != SUBSETS.end(),
            ERR_BAD_REQUEST,
            "wrong subset: " << subset);

        auto branch = getBranch(ctx, parameters);
        auto commitId = getCommitId(ctx, parameters, branch);
        bool tested = parameters.getValue<bool>("tested", true);

        auto& txnCore = ctx.txnCore();

        auto query = Query()
            .insertInto(TABLE_TASK)
            .columns("id, " + FIELDS_TASK)
            .values(Query()
                << task.id() << ','
                << commitId << ','
                << txnCore.quote(subset) << ','
                << branch.id() << ','
                << (tested ? "true" : "false"))
            .returning(FIELDS_TASK);

        auto rows = txnCore.exec(query.str());
        ASSERT(rows.size() == 1);

        task.setContext(std::make_unique<Context>(rows[0]));
    }

    json::Value launchParameters(const Task& task) const override
    {
        const auto& ctx = task.context<Context>();

        return json::Value{
            { TYPE, json::Value{TASK_TYPE} },
            { TASK_ID, json::Value{task.id()} },
            { "branch", json::Value{std::to_string(ctx.branchId)} },
            { COMMIT_ID, json::Value{ctx.commitId} },
            { "subset", json::Value{ctx.subset} },
            { "tested", json::Value{ctx.tested} }
        };
    }
};

const auto module = TaskModuleRegistry::get().registerModule<Module>();

} // namespace export_
} // namespace modules
} // namespace tasks_ng
} // namespace wiki
} // namespace maps
