import sqlalchemy as sa
import simplejson

from flask import abort
from lxml import etree as ET

from yandex.maps.wiki import db, config as service_config, fastcgihelpers as fh
from yandex.maps.wiki.pgpool3 import get_pgpool
from yandex.maps.wiki.tasks import EM, register_task_type, grinder
from yandex.maps.wiki.tasks.enums import ExportSubsetType
from yandex.maps.wiki.tasks.models import Base, Task
from yandex.maps.wiki.utils import require

from maps.wikimap.mapspro.libs.python import acl, revision

import six
if six.PY3:
    long = int

TASK_NAME_EXPORT = 'export'

ACL_PATH = 'mpro/tasks/export'


def create_grinder_gateway():
    return grinder.GrinderGateway(service_config.get_config().grinder_params.host)


@register_task_type(name=TASK_NAME_EXPORT)
class Export:
    @staticmethod
    def capabilities_ET():
        return EM.export_task_type(
            EM.subsets(
                EM.subset(ExportSubsetType.SERVICE.value),
                EM.subset(ExportSubsetType.DOMAIN.value),
                EM.subset(ExportSubsetType.MASSTRANSIT.value),
                EM.subset(ExportSubsetType.MRC.value),
                EM.subset(ExportSubsetType.MRC_PEDESTRIAN.value)))

    @staticmethod
    def create(uid, request):
        require(acl.is_permission_granted(get_pgpool(db.CORE_DB), ACL_PATH, uid),
                fh.ServiceException('forbidden', status='ERR_FORBIDDEN'))

        task = ExportTask()
        task.on_create(uid)

        branch = request.values.get('branch', 0)
        branch_id = 0 if str(branch) == 'trunk' else long(branch)
        task.branch_id = branch_id

        commit_id = long(request.values.get('commit', 0))
        if commit_id == 0:
            rgateway = revision.create_gateway(db.CORE_DB, branch_id)
            commit_id = rgateway.head_commit_id()
        task.commit_id = commit_id

        if request.values['subset'] not in ExportSubsetType.values():
            abort(400, "wrong 'subset' parameter value")
        task.subset = request.values['subset']

        if ('experiment' in request.values) and request.values['experiment']:
            task.experiment = request.values['experiment']

        task.tested = (request.values.get('tested', '1') != '0')

        return task

    @staticmethod
    def launch(session, task_id, request):
        task = session.query(ExportTask).get(task_id)
        args = {
            'type': TASK_NAME_EXPORT,
            'taskId': task.id,
            'branch': str(task.branch_id),
            'commitId': task.commit_id,
            'subset': task.subset.value,
            'experiment': task.experiment if task.experiment else '',
            'tested': task.tested
        }

        gateway = create_grinder_gateway()
        return gateway.submit(args)


class ExportResult(Base):
    __tablename__ = 'export_result'
    __table_args__ = (
        sa.PrimaryKeyConstraint('task_id', 'dataset_id'),
        {'schema': 'service'}
    )

    task_id = sa.Column(sa.BigInteger,
                        sa.ForeignKey('service.export_task.id'))
    message = sa.Column(sa.String, nullable=False)
    dataset_id = sa.Column(sa.String)


class ExportTask(Task):
    __tablename__ = 'export_task'
    __table_args__ = {'schema': 'service'}
    __mapper_args__ = {'polymorphic_identity': 'export'}

    id = sa.Column(sa.Integer,
                   sa.ForeignKey('service.task.id'),
                   primary_key=True)
    branch_id = sa.Column(sa.BigInteger)
    commit_id = sa.Column(sa.BigInteger)
    subset = sa.Column(ExportSubsetType.db_type())
    experiment = sa.Column(sa.String)
    tested = sa.Column(sa.Boolean)

    def __result_ET(self):
        ret = EM.export_result()
        task_result = {}

        if self.grinder_task_id is not None:
            with db.get_read_session(db.CORE_DB) as session:
                for r in session.query(ExportResult).filter_by(task_id=self.id):
                    jsonRes = simplejson.loads(r.message)
                    for key, val in six.iteritems(jsonRes):
                        task_result[key] = val

        for key, value in six.iteritems(task_result):
            ret.append(EM.uri(value, description=key))

        return ret

    def __context_ET(self):
        return EM.export_context(
            EM.branch(self.branch_id),
            EM.commit_id(self.commit_id) if self.commit_id is not None else "",
            EM.subset(self.subset.value),
            EM.experiment(self.experiment) if self.experiment else "",
            EM.tested(self.tested))

    def context_ET_brief(self, *args, **kwargs):
        return self.__context_ET()

    def context_ET_full(self, *args, **kwargs):
        return self.__context_ET()

    def result_ET_full(self, page=1, per_page=10, *args, **kwargs):
        return self.__result_ET()

    def change_parameters(self, session, request):
        ns = '{http://maps.yandex.ru/mapspro/tasks/1.x}'

        root = ET.fromstring(request.data)
        if root.tag != ns + 'parameters':
            abort(404, 'wrong root element')

        parameter = root.find(ns + 'parameter')
        if parameter.attrib['key'] != 'tested':
            abort(404, 'wrong parameter element')

        tested = (parameter.attrib['value'] != '0')

        dataset_id = ''
        for r in session.query(ExportResult).filter_by(task_id=self.id):
            if r.dataset_id is not None:
                dataset_id = r.dataset_id

                # update export_task table
                self.tested = tested

                # update export_meta table
                upd = sa.text("UPDATE mds_dataset.export_meta SET tested = :tested WHERE id = :id")
                params = {'tested': tested, "id" : dataset_id}
                session.execute(upd, params)

        return EM.parameters(EM.parameter(key='tested', value=self.tested))
