# coding: utf8

import logging
import math

import sqlalchemy as sa
import geoalchemy2 as ga
import geoalchemy2.shape
from shapely import wkb
from shapely.geometry import mapping
from lxml import etree as ET
import simplejson  # does not cast to unicode strings

from yandex.maps.wiki import utils, db, fastcgihelpers as fh
from yandex.maps.wiki import config
from yandex.maps.wiki.pgpool3 import get_pgpool
from yandex.maps.wiki.utils import string_to_bool

from yandex.maps.wiki.tasks import EM, register_task_type, TASKS_NAMESPACE, grinder
from yandex.maps.wiki.tasks.models import Task, Base
from yandex.maps.wiki.tasks import groupedit_utils as ge_utils

from maps.wikimap.mapspro.libs.python import acl
from maps.wikimap.mapspro.libs.python import geolocks
from maps.wikimap.mapspro.libs.python import revision as rev


NAMESPACES = {'mpt': TASKS_NAMESPACE}

TASK_NAME = 'groupedit'

MAX_MOVE_DISTANCE = 250  # mercator meters, MAPSPRO-1532

ACL_PATH = "mpro/tasks/group-edit"


def create_grinder_gateway():
    return grinder.GrinderGateway(config.get_config().grinder_params.host)


def make_grinder_task_args(task):
    args = {
        'taskId': task.id,
        'type': TASK_NAME,
        'action': task.action,
        'branchId': task.branch_id,
        'commitId': task.commit_id,
        'lockId': task.lock_id,
        'uid': task.created_by,
        'params': task.params
    }

    if task.filter_expression_id is not None:
        args['filterId'] = task.filter_expression_id

    if task.aoi is not None:
        shape = ga.shape.to_shape(task.aoi)
        args['aoi'] = mapping(shape)

    return args


class DummyLock:
    def __init__(self, pgpool, branch_id):
        self.id = 0
        self.branch_id = branch_id
        self.commit_id = self._get_commit_id(pgpool)

    def _get_commit_id(self, pgpool):
        return rev.RevisionsGateway(pgpool, self.branch_id).head_commit_id()

    @classmethod
    def try_lock(cls, pgpool, branch_id):
        return cls(pgpool, branch_id)

    def unlock(self, pgpool):
        pass


class IAction(object):
    def __init__(self, **params):
        self.params = params

    @classmethod
    def from_xml(cls, node):
        raise NotImplementedError

    def permission_id(self):
        raise NotImplementedError


class MoveAction(IAction):
    @classmethod
    def from_xml(cls, node):
        dx = float(node.find("mpt:dx", namespaces=NAMESPACES).text)
        dy = float(node.find("mpt:dy", namespaces=NAMESPACES).text)
        partial_polygons_node = node.find("mpt:partial-polygons", namespaces=NAMESPACES)
        if partial_polygons_node is not None:
            partial_polygons = string_to_bool(partial_polygons_node.text)
        else:
            partial_polygons = False

        if math.sqrt(dx*dx + dy*dy) > MAX_MOVE_DISTANCE:
            raise fh.ServiceException(
                'move distance too big, dx: %s, dy:%s' % (dx, dy),
                status='ERR_MOVE_MAX_DISTANCE')
        category_ids = []
        for val_node in node.findall(
                'mpt:category-ids/mpt:category-id', namespaces=NAMESPACES):
            category_ids.append(val_node.text)
        return cls(dx=dx, dy=dy, categoryIds=category_ids, partialPolygons=partial_polygons)

    def permission_id(self):
        return 'move'


class UpdateAttrsAction(IAction):
    @classmethod
    def from_xml(cls, node):
        attributes = {}
        for attr_node in node.findall('mpt:attribute', namespaces=NAMESPACES):
            values = []
            for val_node in attr_node.findall(
                    'mpt:values/mpt:value', namespaces=NAMESPACES):
                val = val_node.text
                val = val if val is None else val.strip()
                if val:
                    values.append(val)
            attributes[attr_node.attrib['id']] = values
        return cls(attributes=attributes)

    def permission_id(self):
        return 'attributes'


class SetStateAction(IAction):
    @classmethod
    def from_xml(cls, node):
        if (node.text != 'deleted'):
            raise fh.ServiceException(
                'only state=deleted is supported', status='ERR_BAD_REQUEST')
        return cls()

    def permission_id(self):
        return 'delete'


ACTIONS = {
    'move': MoveAction,
    'attributes': UpdateAttrsAction,
    'state': SetStateAction,
}

EXCLUDED_CAT_IDS = ['feed_region', 'merge_region']


@register_task_type(name=TASK_NAME)
class GroupEdit:
    @staticmethod
    def capabilities_ET():
        cfg_et = ET.parse(config.get_config().editor_params.config)
        cfg_et.xinclude()

        cat_ids_et = cfg_et.xpath('categories/category')
        cat_ids = []
        for cat_id_et in cat_ids_et:
            cat_id = cat_id_et.get('id')
            template_id = cat_id_et.get('template-id')
            if (cat_id not in EXCLUDED_CAT_IDS and
                    cfg_et.xpath('category-templates/category-template[@id="' + template_id + '"]/geometry') and
                    not cfg_et.xpath('topology-groups/topology-group/junction[@category-id="' + cat_id + '"]')):
                cat_ids.append(cat_id)

        ret = EM.groupedit_task_type(
            EM.actions(*[EM.action(id=action) for action in ACTIONS]))

        cat_ids_et = ret.xpath('/groupedit-task-type/actions/action[@id="move"]')[0]
        cat_ids_et.append(EM.category_ids(*[EM.category_id(id) for id in cat_ids]))
        return ret

    @staticmethod
    def create(uid, request):
        task = GroupEditTask()
        task.on_create(uid)

        try:
            task.branch_id = int(request.args.get('branch', 0))

            xml = ET.parse(request.stream)

            aoi_wkb = None
            aoi_elem = xml.find('/mpt:request-groupedit/mpt:aoi/mpt:geometry',
                                namespaces=NAMESPACES)
            if aoi_elem is not None and aoi_elem.text:
                aoi_wkb = utils.geojson_to_mercator_wkb(aoi_elem.text)
                shape = wkb.loads(aoi_wkb)
                if not shape.is_valid:
                    raise fh.ServiceException('invalid geometry',
                                              status='ERR_TOPO_INVALID_GEOMETRY')

                task.aoi = ga.shape.from_shape(shape, srid=3395)

            filter_expr_node = xml.find(
                '/mpt:request-groupedit/mpt:filter-expression', namespaces=NAMESPACES)
            if filter_expr_node is not None:
                task.filter_expression_id = int(filter_expr_node.attrib['id'])

            params_node = xml.find('/mpt:request-groupedit/mpt:params',
                                   namespaces=NAMESPACES)[0]

            action_tag = params_node.tag
            tasks_nsprefix = '{%s}' % TASKS_NAMESPACE
            if action_tag.startswith(tasks_nsprefix):
                action_tag = action_tag[len(tasks_nsprefix):]

            task.action = action_tag
            action = ACTIONS[action_tag].from_xml(params_node)
            task.params = simplejson.dumps(action.params)
            if not acl.is_permission_granted(
                    get_pgpool(db.CORE_DB),
                    ACL_PATH + '/' + action.permission_id(), uid, aoi_wkb):
                raise fh.ServiceException('forbidden', status='ERR_FORBIDDEN')

            if action_tag == "move" and action.params["partialPolygons"]:
                if not acl.is_permission_granted(
                        get_pgpool(db.CORE_DB), "mpro/settings/partial-polygons-move", uid, None):
                    raise fh.ServiceException('forbidden', status='ERR_FORBIDDEN')

        except fh.ServiceException:
            raise
        except Exception:
            logging.exception('invalid request')
            raise fh.ServiceException('error while parsing request',
                                      status='ERR_BAD_REQUEST')

        if aoi_wkb is not None:
            lock = geolocks.try_lock(get_pgpool(db.CORE_DB), uid, task.branch_id, aoi_wkb)
        else:
            lock = DummyLock.try_lock(get_pgpool(db.CORE_DB), task.branch_id)

        if lock is None:
            raise fh.ServiceException('could not lock',
                                      status='ERR_LOCKED')
        task.lock_id = lock.id
        task.commit_id = lock.commit_id

        return task

    @staticmethod
    def launch(session, task_id, request):
        task = ge_utils.get_detached_task(GroupEditTask, task_id)
        pgpool = get_pgpool(db.CORE_DB)

        with ge_utils.StageGuard(task, pgpool) as guard:
            guard.unlock_on_success = False
            guard.next_stage('submitting task')

            gateway = create_grinder_gateway()
            return gateway.submit(make_grinder_task_args(task))


class GroupEditCommit(Base):
    __tablename__ = 'groupedit_commit'
    __table_args__ = {'schema': 'service'}

    task_id = sa.Column(sa.BigInteger,
                        sa.ForeignKey('service.groupedit_task.id'),
                        primary_key=True)
    commit_id = sa.Column(sa.BigInteger, primary_key=True)
    synchronized = sa.Column(sa.Boolean, nullable=False)

    task = sa.orm.relation('GroupEditTask',
                           backref=sa.orm.backref('commits', lazy='dynamic'))


class GroupEditTask(Task):
    __tablename__ = 'groupedit_task'
    __table_args__ = {'schema': 'service'}
    __mapper_args__ = {'polymorphic_identity': 'groupedit'}

    id = sa.Column(sa.Integer,
                   sa.ForeignKey('service.task.id'),
                   primary_key=True)
    branch_id = sa.Column(sa.BigInteger, nullable=False)
    commit_id = sa.Column(sa.BigInteger, nullable=False)
    action = sa.Column(sa.String, nullable=False)

    aoi = sa.Column(ga.Geometry('POLYGON', srid=3395))
    lock_id = sa.Column(sa.BigInteger, nullable=False)
    filter_expression_id = sa.Column(sa.BigInteger)
    params = sa.Column(sa.String, nullable=False)

    @property
    def revocable(self):
        return False

    def resume(self):
        gateway = create_grinder_gateway()
        result = gateway.submit(make_grinder_task_args(self))
        self.grinder_task_id = result.id

    def context_ET_brief(self, *args, **kwargs):
        return EM.groupedit_context(
            EM.branch(self.branch_id),
            EM.commit_id(self.commit_id),
            EM.action(self.action))

    def context_ET_full(self, *args, **kwargs):
        ret = self.context_ET_brief(*args, **kwargs)
        ret.append(EM.lock_id(self.lock_id))
        if self.aoi is not None:
            ret.append(EM.aoi(
                EM.geometry(simplejson.dumps(
                    utils.geoalchemy_to_geojson(self.aoi)))))
        if self.filter_expression_id:
            ret.append(EM.filter_expression(id=self.filter_expression_id))
        ret.append(EM.params(simplejson.dumps(self.params)))
        return ret
