import os.path
import regex
import sqlalchemy as sa

from yandex.maps.wiki import fastcgihelpers as fh
from yandex.maps.wiki import config
from yandex.maps.wiki.tasks import EM, register_task_type, grinder
from yandex.maps.wiki.tasks.models import Base, Task
from yandex.maps.wiki.utils import require, string_to_bool

TASK_NAME = 'import'

ACTION_ADD = 'add'
ACTION_EDIT = 'edit'
ACTION_DELETE = 'delete'


def secure_filename(filename):
    """
    Inspired by werkzeug.utils.secure_filename
    """
    filename = regex.sub(r'\p{C}', '', filename)  # remove all control and zero chars
    filename = filename.replace(os.path.sep, ' ')
    filename = '_'.join(filename.split()).strip('._')
    return filename


def create_grinder_gateway():
    return grinder.GrinderGateway(config.get_config().grinder_params.host)


def message_ET(message):
    element = EM.message(
        description=message.description)

    if message.feature_id is not None:
        element.set("feature-id", str(message.feature_id))

    if message.layer is not None:
        element.set("layer", message.layer)

    return element


@register_task_type(name=TASK_NAME)
class Import:
    @staticmethod
    def capabilities_ET():
        return EM.import_task_type()

    @staticmethod
    def create(uid, request):
        task = ImportTask()
        task.on_create(uid)

        task.branch_id = 0

        task.action = request.values.get('action', ACTION_ADD)
        require(task.action in [ACTION_ADD, ACTION_EDIT, ACTION_DELETE],
                fh.ServiceException('Unknown action: ' + str(task.action),
                                    status='ERR_BAD_REQUEST'))

        task.dry_run = string_to_bool(request.values.get('dry-run', 'false'))

        require(request.files.values(),
                fh.ServiceException('File is not posted',
                                    status='ERR_IMPORT_FILE_NOT_POSTED'))
        uploaded_file = request.files.values()[0]

        task.filename = secure_filename(uploaded_file.filename)
        require(task.filename != '',
                fh.ServiceException('Bad file name',
                                    status='ERR_IMPORT_BAD_FILE_NAME'))

        _, file_ext = os.path.splitext(task.filename)
        require(file_ext in ['.zip', '.gz'],
                fh.ServiceException('Bad file name',
                                    status='ERR_IMPORT_BAD_FILE_NAME'))
        task.file = uploaded_file.read()

        return task

    @staticmethod
    def launch(session, task_id, request):
        task = session.query(ImportTask).get(task_id)
        gateway = create_grinder_gateway()
        return gateway.submit(task.get_grinder_args())


class ImportMessage(Base):
    __tablename__ = 'import_messages'
    __table_args__ = {'schema': 'service'}

    id = sa.Column(sa.BigInteger, primary_key=True)
    task_id = sa.Column(sa.BigInteger,
                        sa.ForeignKey('service.import_task.id'))
    description = sa.Column(sa.String)
    feature_id = sa.Column(sa.BigInteger)
    layer = sa.Column(sa.String)


class ImportTask(Task):
    __tablename__ = 'import_task'
    __table_args__ = {'schema': 'service'}
    __mapper_args__ = {'polymorphic_identity': 'import'}

    id = sa.Column(sa.Integer,
                   sa.ForeignKey('service.task.id'),
                   primary_key=True)
    branch_id = sa.Column(sa.BigInteger)
    action = sa.Column(sa.String)
    filename = sa.Column(sa.String)
    file = sa.orm.deferred(sa.Column(sa.types.LargeBinary))
    result_url = sa.Column(sa.Text)
    dry_run = sa.Column(sa.Boolean)

    def __context_ET(self):
        return EM.import_context(
            EM.branch(self.branch_id),
            EM.filename(self.filename),
            EM.action(self.action),
            EM.dry_run(self.dry_run))

    def context_ET_brief(self, *args, **kwargs):
        return self.__context_ET()

    def result_ET_brief(self, *args, **kwargs):
        session = sa.orm.session.Session.object_session(self)

        total_count = session.query(ImportMessage). \
            filter(ImportMessage.task_id == self.id).count()

        ret = EM.import_result(
            EM.messages(total_count=total_count))

        if self.result_url is not None:
            ret.append(EM.url(self.result_url))
        return ret

    def result_ET_full(self, page=1, per_page=10, *args, **kwargs):
        session = sa.orm.session.Session.object_session(self)

        total_count = session.query(ImportMessage). \
            filter(ImportMessage.task_id == self.id).count()

        per_page = int(per_page)
        page = fh.correct_page(page, per_page, total_count)
        offset = (page - 1) * per_page

        query = session.query(ImportMessage). \
            filter(ImportMessage.task_id == self.id). \
            order_by(ImportMessage.id). \
            offset(offset). \
            limit(per_page)

        ret = EM.import_result(
            EM.messages(
                total_count=total_count,
                page=page,
                per_page=per_page,
                *[message_ET(row) for row in query.all()]))

        if self.result_url is not None:
            ret.append(EM.url(self.result_url))
        return ret

    def get_grinder_args(self):
        return {
            'taskId': self.id,
            'type': TASK_NAME,
            'action': self.action,
            'dryRun': self.dry_run,
            'uid': self.created_by
        }

    def resume(self):
        gateway = create_grinder_gateway()
        result = gateway.submit(self.get_grinder_args())
        self.grinder_task_id = result.id
