# coding: utf8

import sqlalchemy as sa

from yandex.maps.wiki import config
from yandex.maps.wiki.tasks import EM
from yandex.maps.wiki.tasks import register_task_type
from yandex.maps.wiki.tasks import grinder
from yandex.maps.wiki.tasks.models import Task
from yandex.maps.wiki.pgpool3 import get_pgpool
from yandex.maps.wiki.utils import require
from yandex.maps.wiki import db
from yandex.maps.wiki import fastcgihelpers as fh

from maps.wikimap.mapspro.libs.python import revision

import six
if six.PY3:
    long = int

TASK_NAME = 'validation_export'


@register_task_type(name=TASK_NAME)
class ValidationExport:
    @staticmethod
    def capabilities_ET():
        return EM.validation_export_task_type()

    @staticmethod
    def create(uid, request):
        task = ValidationExportTask()
        task.on_create(uid)

        branch_id = int(request.values.get('branch', 0))
        if branch_id == 0:
            branch_mgr = revision.BranchManager(get_pgpool(db.CORE_DB))
            branches = branch_mgr.branches_by_type(revision.BRANCH_TYPE_STABLE, 1)
            require(len(branches) == 1,
                    fh.ServiceException('Wrong stable branch count %d' % len(branches),
                    status='INTERNAL_ERROR'))
            branch_id = branches[0].id
        task.branch_id = branch_id

        commit_id = long(request.values.get('commit', 0))
        if commit_id == 0:
            revision_gateway = revision.create_gateway(db.CORE_DB, branch_id)
            commit_id = revision_gateway.head_commit_id()
        task.commit_id = commit_id

        return task

    @staticmethod
    def launch(session, task_id, request):
        task = session.query(ValidationExportTask).get(task_id)
        args = {
            'type': TASK_NAME,
            'taskId': task.id,
            'uid': task.created_by,
            'branchId': task.branch_id,
            'commitId': task.commit_id
        }
        gateway = grinder.GrinderGateway(config.get_config().grinder_params.host)
        return gateway.submit(args)


class ValidationExportTask(Task):
    __tablename__ = 'validation_export_task'
    __table_args__ = {'schema': 'service'}
    __mapper_args__ = {'polymorphic_identity': TASK_NAME}

    id = sa.Column(sa.Integer, sa.ForeignKey('service.task.id'), primary_key=True)
    branch_id = sa.Column(sa.BigInteger)
    commit_id = sa.Column(sa.BigInteger)
    result = sa.Column(sa.String)

    def context_ET_brief(self, *args, **kwargs):
        ret = EM.validation_export_context(
            EM.branch(self.branch_id),
            EM.commit_id(self.commit_id))
        if self.result:
            ret.append(EM.result(self.result))
        return ret
