import sqlalchemy as sa

from yandex.maps.wiki import db, fastcgihelpers as fh
from yandex.maps.wiki.pgpool3 import get_pgpool
from yandex.maps.wiki import config
from yandex.maps.wiki.tasks import EM, register_task_type, grinder
from yandex.maps.wiki.tasks.models import Task

from maps.wikimap.mapspro.libs.python import revision

import six
if six.PY3:
    long = int

TASK_NAME = 'vrevisions_refresh'

ACTION_REFRESH = 'refresh'
ACTION_CREATE_STABLE = 'create-stable'

ARCHIVE_BRANCH_MAX = 7


def create_grinder_gateway():
    return grinder.GrinderGateway(config.get_config().grinder_params.host)


class BranchStateChecker:
    def __init__(self, id, state):
        self.id = id
        self.state = state

    def check_normal(self):
        self.check_not_locked()
        self.check_not_progress()
        self.check_not_unavailable()

    def check_not_locked(self):
        if self.state == revision.BRANCH_STATE_LOCKED:
            raise fh.ServiceException(
                'Branch {0} locked'.format(self.id),
                status='ERR_BRANCH_LOCKED')

    def check_not_progress(self):
        if self.state == revision.BRANCH_STATE_PROGRESS:
            raise fh.ServiceException(
                'Branch {0} in progress mode'.format(self.id),
                status='ERR_BRANCH_IN_PROGRESS')

    def check_not_unavailable(self):
        if self.state == revision.BRANCH_STATE_UNAVAILABLE:
            raise fh.ServiceException(
                'Branch {0} unavailable'.format(self.id),
                status='ERR_BRANCH_UNAVAILABLE')


class BranchManager:
    def __init__(self):
        self.branch_mgr = revision.BranchManager(get_pgpool(db.CORE_DB))

    def approved_branch_id(self):
        branch_id = self.branch_mgr.branch_id_by_type(revision.BRANCH_TYPE_APPROVED)
        if branch_id is None:
            raise fh.ServiceException(
                'Approved branch not exists', status='ERR_BRANCH_NOT_EXISTS')
        return branch_id

    def optional_stable_branch_id(self):
        return self.branch_mgr.branch_id_by_type(revision.BRANCH_TYPE_STABLE)

    def archive_branch_ids(self):
        return self.branch_mgr.branch_ids_by_type(revision.BRANCH_TYPE_ARCHIVE)

    def check_normal(self, branch_id):
        branch_state = self.branch_mgr.branch_state(branch_id)
        BranchStateChecker(branch_id, branch_state).check_normal()

    def set_progress(self, branch_id, source_branch_state=''):
        prev_branch_state = self.branch_mgr.set_branch_state_restricted(
            branch_id, revision.BRANCH_STATE_PROGRESS, source_branch_state)
        checker = BranchStateChecker(branch_id, prev_branch_state)
        checker.check_not_locked()
        checker.check_not_progress()
        return prev_branch_state

    def set_stable_branch_creation_flag(self, branch_id):
        branch_state = self.branch_mgr.branch_state(branch_id)
        checker = BranchStateChecker(branch_id, branch_state)
        checker.check_not_progress()
        checker.check_not_unavailable()
        self.branch_mgr.concatenate_attrs(branch_id, {'stable_branch_creation': '1'})


@register_task_type(name=TASK_NAME)
class VrevisionsRefresh:
    @staticmethod
    def capabilities_ET():

        def stage_to_ET(stage):
            return EM.stage(id=stage.id,
                             *[stage_to_ET(child) for child in stage.children])

        return EM.vrevisions_refresh_task_type(
            EM.actions(
                EM.action(id=ACTION_REFRESH),
                EM.action(id=ACTION_CREATE_STABLE)
            ),
            *[stage_to_ET(child) for child in config.get_config().vrevisions_refresh_params.stages])

    @staticmethod
    def create(uid, request):
        task = VrevisionsRefreshTask()
        task.on_create(uid)

        task.stages = request.values.get('stages', 'all')
        task.category_groups = request.values.get('category-groups', 'all')
        task.is_exclusive = int(request.values.get('exclusive', '0')) > 0

        task.action = request.values.get('action', ACTION_REFRESH)
        if task.action == ACTION_CREATE_STABLE:
            task.prepare_action_create_stable()
        elif task.action == ACTION_REFRESH:
            branch = request.values.get('branch', revision.TRUNK_BRANCH_ID)
            branch_id = revision.TRUNK_BRANCH_ID if str(branch) == 'trunk' else long(branch)

            if not task.stages:
                raise fh.ServiceException(
                    'Refresh stages missing', status='ERR_MISSING_REFRESH_STAGES')
            if not task.category_groups:
                raise fh.ServiceException(
                    'Refresh category groups missing', status='ERR_MISSING_REFRESH_CATEGORIES')

            task.prepare_action_refresh(branch_id)
        else:
            raise fh.ServiceException(
                'Unknown action: ' + str(task.action), status='ERR_BAD_REQUEST')
        return task

    @staticmethod
    def launch(session, task_id, request):
        task = session.query(VrevisionsRefreshTask).get(task_id)
        args = {
            'taskId': task_id,
            'type': TASK_NAME,
            'action': task.action,
            'branchId': task.branch_id,
            'stages': task.stages,
            'categoryGroups': task.category_groups,
            'isExclusive': int(task.is_exclusive),
            'uid': int(request.values['uid'])
        }

        gateway = create_grinder_gateway()
        return gateway.submit(args)


class VrevisionsRefreshTask(Task):
    __tablename__ = 'vrevisions_refresh_task'
    __table_args__ = {'schema': 'service'}
    __mapper_args__ = {'polymorphic_identity': 'vrevisions_refresh'}

    id = sa.Column(sa.Integer,
                   sa.ForeignKey('service.task.id'),
                   primary_key=True)
    branch_id = sa.Column(sa.BigInteger)
    commit_id = sa.Column(sa.BigInteger)
    action = sa.Column(sa.String)
    stages = sa.Column(sa.String)
    category_groups = sa.Column(sa.String)
    is_exclusive = sa.Column(sa.Boolean)

    @property
    def revocable(self):
        return False  # MAPSPRO-1224

    def __context_ET(self):
        return EM.vrevisions_refresh_context(
            EM.action(self.action),
            EM.branch(self.branch_id),
            EM.commit_id(self.commit_id))

    def context_ET_brief(self, *args, **kwargs):
        return self.__context_ET()

    def context_ET_full(self, *args, **kwargs):
        ret = self.__context_ET()
        if self.category_groups:
            ret.append(EM.category_groups(
                *[EM.category_group(id=group) for group in self.category_groups.split(',')]))
        if self.stages:
            ret.append(EM.stages(
                *[EM.stage(id=a) for a in self.stages.split(',')]))
        return ret

    def prepare_action_create_stable(self):
        branch_manager = BranchManager()
        if branch_manager.optional_stable_branch_id() is not None:
            raise fh.ServiceException(
                'Stable branch already exists', status='ERR_BRANCH_STABLE_ALREADY_EXISTS')
        if len(branch_manager.archive_branch_ids()) >= ARCHIVE_BRANCH_MAX:
            raise fh.ServiceException(
                'Archive branches too many', status='ERR_BRANCH_ARCHIVE_TOO_MANY')
        approved_branch_id = branch_manager.approved_branch_id()
        branch_manager.set_stable_branch_creation_flag(approved_branch_id)
        prev_state = branch_manager.set_progress(approved_branch_id, revision.BRANCH_STATE_NORMAL)
        BranchStateChecker(approved_branch_id, prev_state).check_not_unavailable()
        self.commit_id = revision.create_gateway(db.CORE_DB, approved_branch_id).head_commit_id()
        self.branch_id = approved_branch_id
        self.is_exclusive = True

    def prepare_action_refresh(self, branch_id):
        if self.is_exclusive:
            BranchManager().set_progress(branch_id)
        else:
            BranchManager().check_normal(branch_id)
        self.commit_id = revision.create_gateway(db.CORE_DB, branch_id).head_commit_id()
        self.branch_id = branch_id
