import mock
import pytest

from yandex.maps.wiki.tasks import states
from yandex.maps.wiki.tasks import grinder
from yandex.maps.wiki.tasks import TASKS_NAMESPACE

import json
from lxml import etree as ET
from shapely.geometry import shape

NAMESPACES = {'ns': TASKS_NAMESPACE}

TASK_NAME = 'reject_feedback'

TEST_UID = 1
TEST_BRANCH_ID = 2
TEST_TASK_ID = 1


@pytest.fixture(scope="function")
def setup_permissions():
    with mock.patch("maps.wikimap.mapspro.services.tasks.fastcgi.modules.reject_feedback.reject_feedback.checkPermission") as is_permission_granted:
        is_permission_granted.return_value = True
        yield is_permission_granted


@pytest.mark.usefixtures('setup_acl_groups')
@pytest.mark.usefixtures('setup_permissions')
@pytest.mark.usefixtures('clean_schema')
@pytest.mark.parametrize('module_name', [TASK_NAME])
def test_reject_feedback_empty_filters(client, xmlschema, grinder_class):
    grinder_instance = grinder_class.return_value
    grinder_instance.submit.return_value = grinder.GrinderTaskId('test_id')
    grinder_instance.task_result.return_value = grinder.Result(states.PENDING, 'message')

    geometry_geojson = """{
        "type": "Polygon",
        "coordinates": [
          [
            [100.0, 0.0],
            [101.0, 0.0],
            [101.0, 1.0],
            [100.0, 1.0],
            [100.0, 0.0]
          ]
        ]
      }"""

    reject_reason = 'spam'

    # Check with only geometry filter provided
    response = client.post('/tasks', data={
        'type': TASK_NAME,
        'uid': TEST_UID,
        'geometry': geometry_geojson,
        'reject-reason': reject_reason
    })
    print('RESPONSE DATA:', response.data)
    assert response.status_code == 200

    root = ET.fromstring(response.data)
    xmlschema.assertValid(root)

    # check reject reason
    actual_reject_reason = root.find("./ns:task/ns:context/ns:reject-feedback-context/ns:rejectReason", NAMESPACES).text
    assert reject_reason == actual_reject_reason

    # check no filters are created
    assert len(root.findall("./ns:task/ns:context/ns:reject-feedback-context/ns:workflows/ns:workflow", NAMESPACES)) == 0
    assert len(root.findall("./ns:task/ns:context/ns:reject-feedback-context/ns:sources/ns:source", NAMESPACES)) == 0
    assert len(root.findall("./ns:task/ns:context/ns:reject-feedback-context/ns:types/ns:type", NAMESPACES)) == 0

    # Check retrieved geometry ALMOST equals provided geometry
    actual_geojson = root.find("./ns:task/ns:context/ns:reject-feedback-context/ns:aoi/ns:geometry", NAMESPACES).text

    actual_geometry = shape(json.loads(actual_geojson))
    assert actual_geometry.is_valid

    geometry = shape(json.loads(geometry_geojson))
    assert geometry.is_valid

    assert geometry.almost_equals(actual_geometry)

    grinder_instance.submit.assert_called_once_with({
        'type': TASK_NAME,
        'taskId': TEST_TASK_ID
    })


@pytest.mark.usefixtures('setup_acl_groups')
@pytest.mark.usefixtures('setup_permissions')
@pytest.mark.usefixtures('clean_schema')
@pytest.mark.parametrize('module_name', [TASK_NAME])
def test_reject_feedback_non_empty_filters(client, xmlschema, grinder_class):
    grinder_instance = grinder_class.return_value
    grinder_instance.submit.return_value = grinder.GrinderTaskId('test_id')
    grinder_instance.task_result.return_value = grinder.Result(states.PENDING, 'message')

    geometry_geojson = """{
        "type": "Polygon",
        "coordinates": [
          [
            [120.0, 0.0],
            [121.0, 0.0],
            [121.0, 1.0],
            [120.0, 1.0],
            [120.0, 0.0]
          ]
        ]
      }"""

    reject_reason = 'no-data'
    workflows = 'feedback,task'
    sources = 'feedback,sources'
    types = 'poi,road'

    response = client.post('/tasks', data={
        'type': TASK_NAME,
        'uid': TEST_UID,
        'geometry': geometry_geojson,
        'reject-reason': reject_reason,
        'workflows': [workflows],
        'sources': [sources],
        'types': [types]
    })
    print('RESPONSE DATA:', response.data)
    assert response.status_code == 200

    root = ET.fromstring(response.data)
    xmlschema.assertValid(root)

    print(response.data)
    new_task_id = root.find("./ns:task", NAMESPACES).attrib["id"]

    response = client.get('/tasks/' + str(new_task_id), data={
        'uid': TEST_UID
    })
    assert response.status_code == 200

    # check reject reason
    actual_reject_reason = root.find("./ns:task/ns:context/ns:reject-feedback-context/ns:rejectReason", NAMESPACES).text
    assert reject_reason == actual_reject_reason

    separator = ','
    # check workflows, sources and types
    actual_workflows_etree_elements = root.findall("./ns:task/ns:context/ns:reject-feedback-context/ns:workflows/ns:workflow", NAMESPACES)
    actual_workflows = separator.join([el.text for el in actual_workflows_etree_elements])
    assert workflows == actual_workflows

    actual_sources_etree_elements = root.findall("./ns:task/ns:context/ns:reject-feedback-context/ns:sources/ns:source", NAMESPACES)
    actual_sources = separator.join([el.text for el in actual_sources_etree_elements])
    assert sources == actual_sources

    actual_types_etree_elements = root.findall("./ns:task/ns:context/ns:reject-feedback-context/ns:types/ns:type", NAMESPACES)
    actual_types = separator.join([el.text for el in actual_types_etree_elements])
    assert types == actual_types

    # Check retrieved geometry ALMOST equals provided geometry
    actual_geojson = root.find("./ns:task/ns:context/ns:reject-feedback-context/ns:aoi/ns:geometry", NAMESPACES).text

    actual_geometry = shape(json.loads(actual_geojson))
    assert actual_geometry.is_valid

    geometry = shape(json.loads(geometry_geojson))
    assert geometry.is_valid

    assert geometry.almost_equals(actual_geometry)
