#include "module.h"
#include <yandex/maps/wiki/validator/check.h>
#include <yandex/maps/wiki/validator/categories.h>

#include "../utils/face_builder.h"
#include "../utils/object_elements_within_aoi.h"
#include "../utils/geom.h"
#include "../utils/geom_iterator.h"

#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/spatial_relation.h>

const double MAX_DISTANCE_TO_ROAD = 2e3; // meters

namespace maps {
namespace wiki {
namespace validator {
namespace checks {

using categories::ADDR;
using categories::AD;
using categories::AD_FC;
using categories::AD_EL;
using categories::RD;
using categories::RD_EL;

namespace {

bool isWithinAdmUnit(
        const AddressPoint::TGeom& geom,
        const AdmUnit* admUnit,
        CheckContext* context)
{
    auto viewAdFc = context->objects<AD_FC>();
    auto viewAdEl = context->objects<AD_EL>();

    bool result = false;

    for (TId faceId : admUnit->faces()) {
        const Face* face = viewAdFc.byId(faceId);
        utils::FaceBuilder faceBuilder(face, viewAdEl);

        if (!faceBuilder.valid()) {
            continue;
        }

        if (geolib3::spatialRelation(
                geolib3::Polygon2(faceBuilder.points()),
                geom,
                geolib3::Contains)) {
            if (face->isInterior()) {
                // There should be no faces within an interior face
                return false;
            }
            result = true;
        }
    }

    return result;
}

} // namespace

VALIDATOR_CHECK_PART( addr_geometry, inside_associated_adm_unit,
        ADDR, AD, AD_FC, AD_EL )
{
    auto viewAd = context->objects<AD>();

    context->objects<ADDR>().visit([&](const AddressPoint* addressPoint)
    {
        const auto& associated = addressPoint->associatedObject();

        if (associated.type != AddressPoint::AssociatedObject::Type::AdmUnit
                || !context->objects<AD>().loaded(associated.id)) {
            return;
        }
        const AdmUnit* admUnit = viewAd.byId(associated.id);

        if (utils::objectFacesWithinAoi<AD>(context, admUnit)
                && !isWithinAdmUnit(addressPoint->geom(), admUnit, context)) {
            context->error(
                "addr-outside-adm-unit",
                addressPoint->geom(),
                { addressPoint->id(), admUnit->id() });
        }
    });
}

VALIDATOR_CHECK_PART( addr_geometry, near_associated_road, ADDR, RD, RD_EL )
{
    auto viewRd = context->objects<RD>();

    context->objects<ADDR>().visit([&](const AddressPoint* addressPoint)
    {
        const auto& associated = addressPoint->associatedObject();

        if (associated.type != AddressPoint::AssociatedObject::Type::Road
                || !viewRd.loaded(associated.id)) {
            return;
        }
        const Road* road = viewRd.byId(associated.id);

        if (!road->elements().empty()
                && utils::objectElementsWithinAoi<RD>(context, road)
                && utils::distance(
                        addressPoint->geom(),
                        utils::geomIteratorRange<RD_EL>(
                            road->elements(),
                            context))
                    * utils::mercatorDistanceRatio(
                        addressPoint->geom())
                    > MAX_DISTANCE_TO_ROAD) {
            context->error(
                "addr-too-far-from-road",
                addressPoint->geom(),
                { addressPoint->id(), road->id() });
        }
    });
}

} // namespace checks
} // namespace validator
} // namespace wiki
} // namespace maps
