#include "module.h"
#include <yandex/maps/wiki/validator/check.h>
#include <yandex/maps/wiki/validator/categories.h>

#include "../utils/names.h"

#include <algorithm>

namespace maps {
namespace wiki {
namespace validator {

namespace utils {
namespace {

template<>
struct GeomForReport<AddressPoint> {
    static geolib3::Point2 geom(const AddressPoint* point) {
        return point->geom();
    }
};

} // namespace utils
} // namespace

namespace checks {

using categories::ADDR;
using categories::ADDR_NM;

namespace {

const icu::RegexPattern ILLEGAL_LETTER_REGEX =
    utils::compilePattern("[ьъый]", UREGEX_CASE_INSENSITIVE);
const icu::RegexPattern LOWERCASE_LETTER_REGEX =
    utils::compilePattern(R"(\p{Lowercase})");

} // namespace

VALIDATOR_SIMPLE_CHECK( addr_naming, ADDR, ADDR_NM )
{
    context->objects<ADDR>().visit([&](const AddressPoint* addrPoint) {
        utils::runBasicNamesCheck<ADDR_NM>(addrPoint, context);

        for (const NameDatum& officialNameDatum
                : utils::officialNames<ADDR_NM>(addrPoint, context)) {
            if (utils::matchesPattern(
                    officialNameDatum.name, LOWERCASE_LETTER_REGEX)) {
                context->warning(
                        "lowercase-letter-in-addr-name",
                        addrPoint->geom(), { addrPoint->id() });
            }
            if (officialNameDatum.lang == LANG_RU) {
                if (utils::matchesPattern(
                        officialNameDatum.name, ILLEGAL_LETTER_REGEX)) {
                    context->error(
                            "illegal-letter-in-addr-name",
                            addrPoint->geom(), { addrPoint->id() });
                }
            }
        }
    });
}

} // namespace checks
} // namespace validator
} // namespace wiki
} // namespace maps
