#include "module.h"

#include <yandex/maps/wiki/validator/check.h>
#include <yandex/maps/wiki/validator/categories.h>

#include "../utils/names.h"

#include <algorithm>

namespace maps::wiki::validator::checks {

using categories::AD;
using categories::AD_SUBST;
using categories::AD_NM;

template<class TAdmUnit>
void checkHiearchyAddrName(const TAdmUnit* admUnit, CheckContext* context)
{
    if (admUnit->levelKind() != AdmUnit::LevelKind::Locality) {
        return;
    }

    auto viewAd = context->objects<AD>();

    const std::set<TId> excludedFromAddress(
        admUnit->addressExclusions().begin(),
        admUnit->addressExclusions().end());
    std::vector<const AdmUnit*> parents;
    std::set<TId> parentIds;
    auto parentId = admUnit->parent();
    while (parentId) {
        if (parentIds.count(parentId)) {
            context->fatal(
                "cycle-in-adm-unit-hierarchy",
                boost::none,
                {admUnit->id()});
            return;
        }
        parentIds.insert(parentId);
        if (!viewAd.loaded(parentId)) {
            context->error(
                "parent-ad-is-outside-of-validation-area",
                boost::none,
                {parentId, admUnit->id()});
            break;
        }
        const auto curAdParent = viewAd.byId(parentId);
        if (!excludedFromAddress.count(parentId)) {
            parents.push_back(curAdParent);
        }
        parentId = curAdParent->parent();
    }

    const auto addrNames = utils::namesByType<AD_NM, TAdmUnit>(
        admUnit, context, NameRelation::Type::AddressLabel);
    for (const auto* parentAd : parents) {
        const auto parentAddrNames = utils::namesByType<AD_NM, AdmUnit>(
            parentAd, context, NameRelation::Type::AddressLabel);
        for (const auto& addrName : addrNames) {
            for (const auto& parentAddrName : parentAddrNames) {
                if (addrName.lang != parentAddrName.lang) {
                    continue;
                }
                if (parentAddrName.name.find(addrName.name) != std::string::npos) {
                    context->critical(
                        "ad-addr-hierarchy-contains-same-name",
                        boost::none,
                        {parentAd->id(), admUnit->id()});
                }
            }
        }
    }
}

VALIDATOR_SIMPLE_CHECK( ad_address, AD, AD_SUBST, AD_NM )
{
    context->objects<AD>().visit([&](const AdmUnit* admUnit) {
        checkHiearchyAddrName(admUnit, context);
    });

    context->objects<AD_SUBST>().visit([&](const AdmUnitSubst* admUnit) {
        checkHiearchyAddrName(admUnit, context);
    });
}

} // namespace maps::wiki::validator::checks
