#include "module.h"
#include <yandex/maps/wiki/validator/check.h>
#include <yandex/maps/wiki/validator/categories.h>

#include "../utils/face_builder.h"
#include "../utils/face_checks.h"
#include "../utils/find_gaps_representatives.h"
#include "../utils/object_elements_within_aoi.h"

#include <maps/libs/geolib/include/convex_hull.h>
#include <maps/libs/geolib/include/polygon.h>

#include <algorithm>
#include <vector>

const double MAX_FACE_DISTANCE = 7e3; // meters

namespace maps {
namespace wiki {
namespace validator {
namespace checks {

using categories::AD;
using categories::AD_FC;
using categories::AD_SUBST;
using categories::AD_SUBST_FC;
using categories::AD_EL;

namespace {

const size_t VISIT_BATCH_SIZE = 100;

geolib3::Polygon2 geomForReport(
        const geolib3::Polygon2& a,
        const geolib3::Polygon2& b)
{
    return geolib3::bufferedConvexHull(
        { a.boundingBox().center(), b.boundingBox().center() },
        utils::BUFFER_DISTANCE);
}

void checkFaceProximity(
        const AdmUnit* admUnit,
        CheckContext* context)
{
    std::vector<geolib3::Polygon2> geoms;

    auto viewAdFc = context->objects<AD_FC>();
    auto viewAdEl = context->objects<AD_EL>();

    for (TId faceId : admUnit->faces()) {
        const Face* face = viewAdFc.byId(faceId);
        utils::FaceBuilder faceBuilder(face, viewAdEl);

        auto geom = utils::facePolygon(faceBuilder);
        if (geom) {
            geoms.push_back(std::move(*geom));
        }
    }

    for (const auto& gapRepresentatives
            : utils::findGapsRepresentatives(geoms, MAX_FACE_DISTANCE)) {
        context->error(
            "faces-too-far",
            geomForReport(
                geoms[gapRepresentatives.first],
                geoms[gapRepresentatives.second]),
            { admUnit->id(),
              admUnit->faces()[gapRepresentatives.first],
              admUnit->faces()[gapRepresentatives.second] });
    }
}

} // namespace

VALIDATOR_CHECK_PART( ad_geometry, nesting_intersections, AD, AD_FC, AD_SUBST, AD_SUBST_FC, AD_EL )
{
    utils::runFaceIntersectionsNestingCheck<AD>(context);
    utils::runFaceIntersectionsNestingCheck<AD_SUBST>(context);
}

VALIDATOR_CHECK_PART( ad_geometry, face_proximity, AD, AD_FC, AD_EL )
{
    context->objects<AD>().batchVisit([&](const AdmUnit* admUnit) {
        if (!utils::objectFacesWithinAoi<AD>(context, admUnit)) {
            return;
        }

        if (admUnit->levelKind() >= AdmUnit::LevelKind::Locality
            && admUnit->levelKind() != AdmUnit::LevelKind::Other) {
            checkFaceProximity(admUnit, context);
        }
    }, VISIT_BATCH_SIZE);
}

} // namespace checks
} // namespace validator
} // namespace wiki
} // namespace maps
