#include "module.h"
#include <yandex/maps/wiki/validator/check.h>
#include <yandex/maps/wiki/validator/categories.h>

#include <yandex/maps/wiki/common/misc.h>

#include "../utils/misc.h"
#include "../utils/names.h"

#include <map>
#include <set>
#include <unordered_set>
#include <utility>

namespace maps {
namespace wiki {
namespace validator {
namespace checks {

using categories::AD;
using categories::AD_SUBST;
using categories::AD_NM;

namespace {

const TIsoCode UN_ISO_CODE = "001";
const std::vector<TIsoCode> UN_RECOGNITION = {UN_ISO_CODE};

typedef std::vector<const AdmUnit*> THierarchyPath;
typedef std::unordered_set<const AdmUnit*> AdmUnitSet;

template <class AdmUnitCategory>
void localCheck(
        const typename AdmUnitCategory::TObject* admUnit,
        CheckContext* context)
{
    if (admUnit->parent()) {
        auto viewAd = context->objects<AD>();

        if (!viewAd.loaded(admUnit->parent())) {
            return;
        }
        const AdmUnit* parent = viewAd.byId(admUnit->parent());

        if (parent->levelKind() > admUnit->levelKind() ||
            admUnit->levelKind() == AdmUnit::LevelKind::Country) {
                context->fatal(
                        "wrong-level-kind-sequence", boost::none,
                        {admUnit->id(), parent->id()});
        }

        if (parent->levelKind() == AdmUnit::LevelKind::Other) {
            context->fatal(
                    "bound-to-ad-level-kind-other", boost::none,
                    { admUnit->id(), parent->id() });
        }
    } else {
        if (!common::isIn(
                    admUnit->levelKind(),
                    { AdmUnit::LevelKind::Country, AdmUnit::LevelKind::Other })) {
            context->fatal(
                    "adm-unit-not-in-hierarchy", boost::none, {admUnit->id()});
        } else if (admUnit->isoCode().empty()) {
            context->fatal(
                    "top-level-ad-without-iso-code",
                    boost::none, {admUnit->id()});
        }
    }
}

class PathChecker
{
    typedef std::vector<std::string> NamePath;

public:
    PathChecker(CheckContext* context)
        : context_(context)
    { }

    void check(const THierarchyPath& path)
    {
        REQUIRE(!path.empty(), "Empty path");

        if (path.front()->isLeaf() && !utils::isNamed<AD_NM>(path.front(), context_)) {
            return;
        }

        std::map<std::string, NamePath> namePathsByLang;
        for (const NameDatum& officialNameDatum
                : utils::officialNames<AD_NM>(path.front(), context_)) {
            namePathsByLang.insert(
                std::make_pair(officialNameDatum.lang, NamePath()));
        }

        for (const AdmUnit* admUnit : path) {
            for (const NameDatum& officialNameDatum
                    : utils::officialNames<AD_NM>(admUnit, context_)) {
                auto langNamePathIt =
                    namePathsByLang.find(officialNameDatum.lang);
                if (langNamePathIt != std::end(namePathsByLang)) {
                    langNamePathIt->second.push_back(officialNameDatum.name);
                }
            }
        }

        for (auto& langNamePathPair : namePathsByLang) {
            auto insertionResult = langNamePathsToIds_.insert(
                std::make_pair(std::move(langNamePathPair), path.front()->id()));
            if (!insertionResult.second) {
                context_->error(
                        "duplicate-hierarchy-path", boost::none,
                        {insertionResult.first->second, path.front()->id()});
            }
        }
    }

private:
    std::map<std::pair<std::string, NamePath>, TId> langNamePathsToIds_;
    CheckContext* context_;
};

// return empty path if path is invalid
// precondition: badPathBeginnings contains all admUnits with invalid parents
THierarchyPath tracePath(
        const AdmUnit* admUnit,
        AdmUnitSet* badPathBeginnings,
        CheckContext* context)
{
    auto viewAd = context->objects<AD>();

    THierarchyPath curPath;
    const AdmUnit* curAdmUnit = admUnit;
    while (curAdmUnit->parent()) {
        if (badPathBeginnings->count(curAdmUnit)) {
            return THierarchyPath{};
        }

        auto prevOccurence = std::find(
                curPath.begin(), curPath.end(), curAdmUnit);
        if (prevOccurence != curPath.end()) {
            std::vector<TId> cycle;
            for (auto it = prevOccurence; it != curPath.end(); ++it) {
                cycle.push_back((*it)->id());
                badPathBeginnings->insert(*it);
            }
            context->fatal("cycle-in-adm-unit-hierarchy", boost::none, cycle);
            return THierarchyPath{};
        }

        curPath.push_back(curAdmUnit);
        curAdmUnit = viewAd.byId(curAdmUnit->parent());
    }
    curPath.push_back(curAdmUnit);
    return curPath;
}

template<class TAdmUnit>
void checkLinkedAttributes(const TAdmUnit* admUnit, CheckContext* context)
{
    if (admUnit->isTown() && admUnit->levelKind() != AdmUnit::LevelKind::Locality) {
        context->error("wrong-town-level-kind", boost::none, {admUnit->id()});
    }

    const bool isSignificantAdmUnitInformal = admUnit->isInformal() && (
        admUnit->levelKind() < AdmUnit::LevelKind::Locality ||
        (admUnit->levelKind() == AdmUnit::LevelKind::Locality && admUnit->isTown())
    );
    if (isSignificantAdmUnitInformal) {
        context->fatal("significant-adm-unit-informal", boost::none, {admUnit->id()});
    }

    if (admUnit->levelKind() != AdmUnit::LevelKind::Locality && admUnit->isMunicipality()) {
        context->error("wrong-municipality-level-kind", boost::none, {admUnit->id()});
    }
}

} // namespace

VALIDATOR_CHECK_PART( ad_hierarchy, common, AD, AD_SUBST, AD_NM )
{
    auto viewAd = context->objects<AD>();
    auto viewAdSubst = context->objects<AD_SUBST>();

    AdmUnitSet badPathBeginnings;

    viewAd.visit([&](const AdmUnit* admUnit) {
        localCheck<AD>(admUnit, context);

        if (admUnit->parent()) {
            if (!viewAd.loaded(admUnit->parent())) {
                badPathBeginnings.insert(admUnit);
            }
        }

        if (!admUnit->isLeaf() && !utils::isNamed<AD_NM>(admUnit, context)) {
            context->fatal(
                "unnamed-adm-unit-not-leaf",
                boost::none, {admUnit->id()});
        }
    });

    PathChecker pathChecker(context);

    viewAd.visit([&](const AdmUnit* admUnit) {
        THierarchyPath path = tracePath(admUnit, &badPathBeginnings, context);
        if (path.empty()) {
            return;
        }

        pathChecker.check(path);
    });

    viewAdSubst.visit([&](const AdmUnitSubst* admUnitSubst) {
        localCheck<AD_SUBST>(admUnitSubst, context);

        const TId gAdId = admUnitSubst->generalHierarchyId();
        const AdmUnit* gAd = viewAd.byId(gAdId);
        if (!gAd->isLeaf() && !utils::isNamed<AD_NM>(admUnitSubst, context)) {
            context->fatal(
                "unnamed-adm-unit-not-leaf",
                boost::none, {admUnitSubst->id()});
        }
    });
}

// recognition checks

namespace {

typedef std::map<TIsoCode, TId> AdIdByIsoCodeMap;
typedef std::map<TId, AdIdByIsoCodeMap> RecognitionByGeneralAdIdMap;

template <class TAdmUnit>
void addRecognition(
    CheckContext* context, const TAdmUnit* admUnit, AdIdByIsoCodeMap& adByIsoCodeMap)
{
    const auto& recognitionIsoCodes = admUnit->recognitionIsoCodes().empty()
        ? UN_RECOGNITION
        : admUnit->recognitionIsoCodes();
    for (const auto& isoCode : recognitionIsoCodes) {
        auto insertResult = adByIsoCodeMap.insert({isoCode, admUnit->id()});
        if (!insertResult.second) {
            context->fatal(
                "duplicate-adm-unit-recognition",
                boost::none,
                {admUnit->id(), insertResult.first->second});
        }
    }
}

} // namespace

VALIDATOR_CHECK_PART( ad_hierarchy, recognition, AD, AD_SUBST )
{
    RecognitionByGeneralAdIdMap recognition;

    context->objects<AD_SUBST>().visit([&](const AdmUnitSubst* admUnitSubst) {
        const TId gAdId = admUnitSubst->generalHierarchyId();
        if (!gAdId || !context->objects<AD>().loaded(gAdId)) {
            return;
        }

        auto it = recognition.find(gAdId);
        if (it == recognition.end()) {
            it = recognition.insert({gAdId, {}}).first;
            addRecognition(context, context->objects<AD>().byId(gAdId), it->second);
        }
        addRecognition(context, admUnitSubst, it->second);
    });

    for (const auto& recognitionMap : recognition) {
        if (!recognitionMap.second.count(UN_ISO_CODE)) {
            context->error(
                "adm-unit-un-recognition-missing",
                boost::none,
                {recognitionMap.first});
        }
    }

    std::set<TId> alternatives;
    context->objects<AD_SUBST>().visit([&](const AdmUnitSubst* admUnit) {
        alternatives.insert(admUnit->generalHierarchyId());
        alternatives.insert(admUnit->id());
    });

    context->objects<AD>().visit([&](const AdmUnit* admUnit) {
        if (!admUnit->recognitionIsoCodes().empty() &&
            !alternatives.contains(admUnit->id()))
        {
            context->fatal("adm-unit-missed-substitution", boost::none, {admUnit->id()});
        }
    });
    context->objects<AD_SUBST>().visit([&](const AdmUnitSubst* admUnit) {
        if (!admUnit->recognitionIsoCodes().empty() &&
            !alternatives.contains(admUnit->id()))
        {
            context->fatal("adm-unit-missed-substitution", boost::none, {admUnit->id()});
        }
    });
}

VALIDATOR_CHECK_PART( ad_hierarchy, linked_attributes, AD, AD_SUBST )
{
    context->objects<AD>().visit([&](const AdmUnit* admUnit) {
        checkLinkedAttributes(admUnit, context);
    });

    context->objects<AD_SUBST>().visit([&](const AdmUnitSubst* admUnit) {
        checkLinkedAttributes(admUnit, context);
    });
}

} // namespace checks
} // namespace validator
} // namespace wiki
} // namespace maps
