#include "module.h"
#include "../utils/object_elements_within_aoi.h"

#include <yandex/maps/wiki/validator/categories.h>
#include <yandex/maps/wiki/validator/check.h>
#include <yandex/maps/wiki/common/constants.h>

#include <maps/libs/geolib/include/algorithm.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/exception.h>
#include <maps/libs/geolib/include/multipolygon.h>
#include <maps/libs/geolib/include/polygon.h>

#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>

#include <util/string/cast.h>

namespace maps {
namespace wiki {
namespace validator {
namespace checks {

using categories::BLD;
using categories::BLD_COMPLEX;

namespace {

/**
 * Try to emulate the process of exporting data from NMAPS to Garden
 */
double reduceDoublePrecision(double value)
{
    json::Builder builder;
    builder.setDoublePrecision(common::JSON_DOUBLE_PRECISION);
    builder << value;

    auto str = builder.str();
    auto json = json::Value::fromString(str);
    return json.as<double>();
}

class ReducePointPrecisionTransform : public geolib3::ICoordTransform2
{
public:
    geolib3::Point2 operator()(
        const geolib3::Point2& point,
        geolib3::TransformDirection direction) const override
    {
        ASSERT(direction == geolib3::TransformDirection::Forward);
        return geolib3::Point2(
            reduceDoublePrecision(point.x()),
            reduceDoublePrecision(point.y()));
    }
};

}

VALIDATOR_CHECK_PART( bld_complex, bld_complex_relations, BLD_COMPLEX, BLD )
{
    context->objects<BLD_COMPLEX>().visit(
        [&](const BuildingComplex* buildingComplex) {
            if (!utils::objectElementsWithinAoi(context, buildingComplex)) {
                return;
            }

            if (buildingComplex->elements().size() < 2) {
                context->error("too-few-elements", boost::none, { buildingComplex->id() });
            }
        });
}

VALIDATOR_CHECK_PART( bld_complex, bld_complex_geometry, BLD, BLD_COMPLEX )
{
    geolib3::SimpleGeometryTransform2 reduceGeometryPrecision(ReducePointPrecisionTransform{});

    context->objects<BLD_COMPLEX>().visit(
        [&](const BuildingComplex* buildingComplex) {
            if (!utils::objectElementsWithinAoi(context, buildingComplex)) {
                return;
            }

            if (buildingComplex->elements().empty()) {
                return;
            }

            std::vector<geolib3::Polygon2> geoPolygons;

            for (auto bldId : buildingComplex->elements()) {
                const auto* bld = context->objects<BLD>().byId(bldId);
                auto geoGeom = geolib3::convertMercatorToGeodetic(bld->geom());
                geoPolygons.push_back(reduceGeometryPrecision(geoGeom, geolib3::TransformDirection::Forward));
            }

            try {
                auto geoMultiPolygon = geolib3::unitePolygons(geoPolygons);
                geolib3::convertGeodeticToMercator(geoMultiPolygon);
            } catch (const geolib3::ValidationError& e) {
                context->fatal("bad-geometry", e.point(), { buildingComplex->id() });
            }
        });
}

} // namespace checks
} // namespace validator
} // namespace wiki
} // namespace maps
