#include "logic_check_common.h"
#include <yandex/maps/wiki/common/rd/lane.h>

#include <algorithm>
#include <map>
#include <vector>

namespace maps::wiki::validator::checks {

using categories::COND_DS;
using categories::COND_DS_EL;
using categories::RD_JC;
using categories::RD_EL;

namespace {

const std::string COND_DS_EL_TYPE_TOPONYM = "toponym";
size_t MAX_CONSECUTIVE_TOPONYMS_COUNT = 3;

template <>
common::AccessId
getCondAccessId<CondDirectionSign>(const CondDirectionSign*)
{
    return common::AccessId::None;
}

} // namespace

VALIDATOR_CHECK_PART( cond_ds_logic, passability, COND_DS, RD_JC, RD_EL )
{
    context->objects<COND_DS>().visit([&](const CondDirectionSign* condDs) {
        conditionPassabilityCheck<COND_DS>(context, condDs, 0, Severity::Critical, Severity::Critical);
    });
}

VALIDATOR_CHECK_PART( cond_ds_logic, consecutive_toponyms, COND_DS, COND_DS_EL, RD_JC )
{
    context->objects<COND_DS>().visit([&](const CondDirectionSign* condDs) {
        std::vector<const CondDirectionSignElement*> condDsEls;

        for (const auto& condDsElId : condDs->condDsElIds()) {
            if (context->objects<COND_DS_EL>().loaded(condDsElId)) {
                condDsEls.push_back(context->objects<COND_DS_EL>().byId(condDsElId));
            } else {
                context->fatal(
                    "cond-ds-element-not-loaded",
                    boost::none,
                    {condDs->id()});
                return;
            }
        }

        std::sort(
            condDsEls.begin(),
            condDsEls.end(),
            [](const CondDirectionSignElement* lhs, const CondDirectionSignElement* rhs) {
                return lhs->order() < rhs->order();
            });

        size_t consecutive_toponyms_count = 0;
        for (const auto& condDsEl : condDsEls) {
            if (condDsEl->newLine()) {
                consecutive_toponyms_count = 0;
            }
            if (condDsEl->type() == COND_DS_EL_TYPE_TOPONYM) {
                ++consecutive_toponyms_count;
            }
            if (consecutive_toponyms_count >= MAX_CONSECUTIVE_TOPONYMS_COUNT) {
                context->critical(
                    "cond-ds-too-many-consecutive-toponyms",
                    context->objects<RD_JC>().byId(condDs->viaJunction())->geom(),
                    {condDs->id()});
                break;
            }
        }
    });
}

VALIDATOR_CHECK_PART( cond_ds_logic, duplicates, COND_DS, RD_JC )
{
    std::map<TId, std::vector<const CondDirectionSign*>> condDsByRdJc;
    context->objects<COND_DS>().visit([&](const CondDirectionSign* condDs) {
        condDsByRdJc[condDs->viaJunction()].push_back(condDs);
    });

    auto viewRdJc = context->objects<RD_JC>();

    auto geomForReport = [&](TId rdJc) -> boost::optional<geolib3::Point2> {
        if (!viewRdJc.loaded(rdJc)) {
            return boost::none;
        }
        return viewRdJc.byId(rdJc)->geom();
    };

    for (const auto& [rdJcId, condDsInRdJc] : condDsByRdJc) {
        if (condDsInRdJc.size() == 1) {
            continue;
        }

        std::map<std::vector<TId>, common::LaneDirection> dirByRdElIds;
        std::map<std::pair<TId, common::LaneDirection>, std::vector<TId>> rdElIdsByFromAndDir;

        std::map<std::vector<TId>, TId> condDsIdByRdElIds;
        std::map<std::pair<TId, common::LaneDirection>, TId> condDsIdByDir;

        for (const auto& condDs : condDsInRdJc) {
            std::vector<TId> rdElIds;
            rdElIds.reserve(condDs->toRoadElements().size() + 1);
            rdElIds.push_back(condDs->fromRoadElement());
            // cond_ds_topo check will ensure that rdElId order in toRoadElements()
            // matches to sequence numbers
            for (const auto& [_, rdElId] : condDs->toRoadElements()) {
                rdElIds.push_back(rdElId);
            }
            {
                const auto iter = dirByRdElIds.find(rdElIds);
                if (iter == dirByRdElIds.end()) {
                    dirByRdElIds[rdElIds] = condDs->direction();
                    condDsIdByRdElIds[rdElIds] = condDs->id();
                } else if (iter->second != condDs->direction()) {
                    context->report<geolib3::Point2>(
                        Severity::Critical,
                        "cond-ds-same-rd_els-different-direction",
                        geomForReport(condDs->viaJunction()),
                        {condDs->viaJunction(), condDs->id(), condDsIdByRdElIds[rdElIds]});
                }
            }
            {
                auto fromAndDir = std::make_pair(condDs->fromRoadElement(), condDs->direction());
                const auto iter = rdElIdsByFromAndDir.find(fromAndDir);
                if (iter == rdElIdsByFromAndDir.end()) {
                    rdElIdsByFromAndDir[fromAndDir] = rdElIds;
                    condDsIdByDir[fromAndDir] = condDs->id();
                } else if (iter->second != rdElIds) {
                    context->report<geolib3::Point2>(
                        Severity::Critical,
                        "cond-ds-same-direction-different-rd_els",
                        geomForReport(condDs->viaJunction()),
                        {condDs->viaJunction(), condDs->id(), condDsIdByDir[fromAndDir]});
                }
            }
        }
    }
}

} // maps::wiki::validator::checks
