#include "logic_check_common.h"

namespace maps {
namespace wiki {
namespace validator {
namespace checks {

using categories::COND_LANE;

VALIDATOR_CHECK_PART( cond_lane_logic, passability, COND_LANE, RD_JC, RD_EL )
{
    context->objects<COND_LANE>().visit([&](const ConditionLane* condition) {
        conditionPassabilityCheck<COND_LANE>(context, condition, 0, Severity::Warning);
    });
}

VALIDATOR_CHECK_PART( cond_lane_logic, access_id, COND_LANE, RD_JC, RD_EL )
{
    std::vector<TId> idsToReport;

    context->objects<COND_LANE>().visit([&](const ConditionLane* condition) {
        conditionAccessIdCheck<COND_LANE>(context, condition, "cond_lane", idsToReport);
    });
}

VALIDATOR_CHECK_PART( cond_lane_logic, lanes, COND_LANE, RD_JC, RD_EL )
{
    context->objects<COND_LANE>().visit(
            [&](const ConditionLane* conditionLane) {
        const auto& roadElements = context->objects<RD_EL>();
        const auto& junctions = context->objects<RD_JC>();

        if (!junctions.loaded(conditionLane->viaJunction())) {
            return;
        }
        if (!roadElements.loaded(conditionLane->fromRoadElement())) {
            return;
        }
        if (conditionLane->laneMin() > conditionLane->laneMax()) {
            context->fatal("cond_lane-invalid-lanes-range",
                junctions.byId(conditionLane->viaJunction())->geom(),
                {conditionLane->id()});
            return;
        }

        const auto fromRoadElement =
                roadElements.byId(conditionLane->fromRoadElement());
        const auto& lanes =
            (conditionLane->viaJunction() == fromRoadElement->startJunction()
             ? fromRoadElement->fromLanes()
             : fromRoadElement->toLanes());

        if (lanes.empty()) {
            context->fatal(
                conditionLane->viaJunction() == fromRoadElement->startJunction()
                ? "cond_lane-element-f_lane-not-set"
                : "cond_lane-element-t_lane-not-set",
                junctions.byId(conditionLane->viaJunction())->geom(),
                {conditionLane->id(), conditionLane->fromRoadElement()});
            return;
        }
        if (conditionLane->laneMin() > lanes.size() - 1 ||
            conditionLane->laneMax() > lanes.size() - 1) {
            context->fatal("cond_lane-element-lanes-out-of-range",
                junctions.byId(conditionLane->viaJunction())->geom(),
                {conditionLane->id(), conditionLane->fromRoadElement()});
            return;
        }
        for (size_t laneNum = conditionLane->laneMin();
            laneNum <= conditionLane->laneMax(); ++laneNum) {
            if (!lanes[laneNum].contains(conditionLane->lane())) {
                context->fatal("cond_lane-element-lanes-not-match",
                    junctions.byId(conditionLane->viaJunction())->geom(),
                    {conditionLane->id(), conditionLane->fromRoadElement()});
            }
        }
    });
}

} // namespace checks
} // namespace validator
} // namespace wiki
} // namespace maps
