#include "component_report.h"
#include "../utils/misc.h"
#include "../utils/road_utils.h"

#include <maps/libs/geolib/include/convex_hull.h>

#include <algorithm>
#include <set>
#include <stack>
#include <unordered_map>
#include <vector>

namespace maps::wiki::validator::checks {

using categories::RD_EL;
using categories::RD_JC;
using categories::COND;

GraphComponent::GraphComponent(
    CheckContext* context,
    const graph::NodeIds& component,
    Type type)
    : onAoiBorder(false)
    , isolated(false)
{
    size_t count = 0;
    auto iter = component.begin();
    while (iter != component.end() &&
        count < GraphComponent::MAX_REPRESENTATIVES_COUNT) {
            representatives.push_back(*iter);
            ++iter;
            ++count;
    }

    if (type == Type::Secondary) {
        auto viewRdEl = context->objects<RD_EL>();

        geolib3::PointsVector allPoints;
        for (auto rdElId : component) {
            const auto& curPoints = viewRdEl.byId(rdElId)->geom().points();
            allPoints.insert(allPoints.end(), curPoints.begin(), curPoints.end());
        }
        geom = geolib3::bufferedConvexHull(allPoints, utils::BUFFER_DISTANCE);
    }
}

bool
calculateBorderFlag(
    CheckContext* context,
    const std::set<TId>& component)
{
    auto viewRdEl = context->objects<RD_EL>();
    auto viewRdJc = context->objects<RD_JC>();

    for (auto rdElId : component) {
        const auto rdEl = viewRdEl.byId(rdElId);
        if (!viewRdJc.loaded(rdEl->startJunction()) ||
            !viewRdJc.loaded(rdEl->endJunction())) {
                return true;
        }
    }
    return false;
}

bool
calculateIsolatedFlag(
    CheckContext* context,
    const std::set<TId>& component,
    utils::RdElFilter graphFilter)
{
    ObjectIdSet sccRdElIds;

    auto viewRdEl = context->objects<RD_EL>();
    auto viewRdJc = context->objects<RD_JC>();

    auto isRdElInScc = [&](TId rdElId) -> bool {
        if (!viewRdEl.loaded(rdElId)) {
            return false;
        }
        auto rdEl = viewRdEl.byId(rdElId);
        if (!sccRdElIds.count(rdElId) && graphFilter(rdEl)) {
            return false;
        }
        return true;
    };

    auto areAllRdElsInScc = [&](const std::vector<TId>& rdElIds) -> bool {
        return std::all_of(rdElIds.begin(), rdElIds.end(), isRdElInScc);
    };

    auto areAllAdjacentRdElsInScc = [&](TId rdJcId) -> bool {
        if (!viewRdJc.loaded(rdJcId)) {
            return false;
        }
        auto rdJc = viewRdJc.byId(rdJcId);
        return areAllRdElsInScc(rdJc->inElements()) && areAllRdElsInScc(rdJc->outElements());
    };

    for (const auto& rdElId : component) {
        sccRdElIds.insert(rdElId);
    }

    return std::all_of(component.begin(), component.end(),
        [&] (const auto& rdElId) {
            auto rdEl = viewRdEl.byId(rdElId);
            return
                areAllAdjacentRdElsInScc(rdEl->startJunction()) &&
                areAllAdjacentRdElsInScc(rdEl->endJunction());
        });
}

StronglyConnectedComponentReport::StronglyConnectedComponentReport(
    CheckContext* context,
    const graph::StronglyConnectedComponents& components,
    utils::RdElFilter graphFilter)
{
    auto maxIter = std::max_element(
        components.begin(), components.end(),
        [](const graph::NodeIds& a, const graph::NodeIds& b) {
            return a.size() < b.size();
        });

    graph::StronglyConnectedComponents secondaries;
    for (auto iter = components.begin(); iter != components.end(); ++iter) {
        if (iter == maxIter) {
            primary_ = GraphComponent(
                context,
                *iter,
                GraphComponent::Type::Primary);
        } else {
            secondaries.push_back(*iter);
        }
    }
    unionSecondaries(context, secondaries, graphFilter);
}

void
StronglyConnectedComponentReport::unionSecondaries(
    CheckContext* context,
    const graph::StronglyConnectedComponents& components,
    utils::RdElFilter graphFilter)
{
    std::unordered_map<TId, size_t> rdElIdToComponentIndex;
    for (size_t i = 0; i < components.size(); ++i) {
        for (auto rdElId : components[i]) {
            rdElIdToComponentIndex[rdElId] = i;
        }
    }

    std::vector<bool> componentsProcessed(components.size(), false);
    for (size_t i = 0; i < components.size(); ++i) {
        if (componentsProcessed[i]) {
            continue;
        }

        std::set<TId> combinedComponent;

        std::stack<TId> dfsStack;
        for (auto rdElId : components[i]) {
            dfsStack.push(rdElId);
        }

        while (!dfsStack.empty()) {
            auto curRdElId = dfsStack.top();
            dfsStack.pop();

            const auto& outRdEls = utils::findRdElContinuations(
                context,
                curRdElId,
                [](const RoadElement*) { return true; },
                utils::RdElContinuationsByJcId(),
                utils::IncidenceDirection::Both,
                [](const RoadElement*) { return true; });

            for (const auto& rdEl : outRdEls) {
                auto outRdElId = rdEl->id();
                auto outIter = rdElIdToComponentIndex.find(outRdElId);
                if (outIter != rdElIdToComponentIndex.end()) {
                    const auto outComponentIndex = outIter->second;
                    if (componentsProcessed[outComponentIndex]) {
                        continue;
                    }
                    for (auto outRdEl : components[outComponentIndex]) {
                        dfsStack.push(outRdEl);
                    }
                    componentsProcessed[outComponentIndex] = true;
                }
            }
            combinedComponent.insert(curRdElId);
        }
        secondaries_.emplace_back(
            context,
            graph::NodeIds(combinedComponent.begin(), combinedComponent.end()),
            GraphComponent::Type::Secondary);

        auto& secondary = secondaries_.back();
        secondary.onAoiBorder = calculateBorderFlag(
            context, combinedComponent);
        secondary.isolated = calculateIsolatedFlag(
            context, combinedComponent, graphFilter);
    }
}

} // namespace maps::wiki::validator::checks
