#include "module.h"
#include <yandex/maps/wiki/validator/check.h>
#include <yandex/maps/wiki/validator/categories.h>

#include "../utils/misc.h"
#include <maps/libs/geolib/include/conversion_geos.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/spatial_relation.h>

#include <geos/geom/Geometry.h>
#include <geos/geom/Polygon.h>
#include <geos/operation/overlay/OverlayOp.h>

#include <unordered_map>
#include <vector>

namespace maps::wiki::validator::checks {

const double MAX_OUTSIDE_RATIO = 0.05;

using categories::ROAD_SURFACE;
using categories::ROAD_MARKING_POLYGONAL;
using categories::ROAD_MARKING_LINEAR;
using categories::ROAD_MARKING_POINT_LANE_DIRECTION;
using categories::ROAD_MARKING_POINT_SYMBOL;
using categories::ROAD_MARKING_POINT_ROAD_SIGN;
using categories::ROAD_MARKING_POINT_ROAD_SIGN_SPEED_LIMIT;
using categories::ROAD_MARKING_POINT_TEXT;

namespace {

geolib3::Polygon2 geomForReport(const RoadMarkingPolygonal* marking)
{
    return marking->geom();
}

geolib3::Point2 geomForReport(const RoadMarkingLinear* marking)
{
    return utils::geomForReport(marking);
}

geolib3::Point2 geomForReport(const RoadMarkingPoint* marking)
{
    return marking->geom();
}

bool
isMajorIntersection(
    const geolib3::Polygon2& roadSurfaceGeom,
    const geolib3::Polygon2& markingGeom)
{
    const auto roadSurfaceGeos = geolib3::internal::geolib2geosGeometry(roadSurfaceGeom);
    const auto markingGeos = geolib3::internal::geolib2geosGeometry(markingGeom);
    const auto intersection = roadSurfaceGeos->intersection(markingGeos.get());
    return (intersection ? intersection->getArea() : 0.0) / markingGeos->getArea() >= MAX_OUTSIDE_RATIO;
}

bool
isMajorIntersection(
    const geolib3::Polygon2& roadSurfaceGeom,
    const geolib3::Polyline2& markingGeom)
{
    const auto roadSurfaceGeos = geolib3::internal::geolib2geosGeometry(roadSurfaceGeom);
    const auto markingGeos = geolib3::internal::geolib2geosGeometry(markingGeom);
    const auto intersection = roadSurfaceGeos->intersection(markingGeos.get());
    return (intersection ? intersection->getLength() : 0.0) / markingGeos->getLength() >= MAX_OUTSIDE_RATIO;
}

bool
isMajorIntersection(
    const geolib3::Polygon2& roadSurfaceGeom,
    const geolib3::Point2& markingGeom)
{
    return geolib3::spatialRelation(
        roadSurfaceGeom, markingGeom, geolib3::Intersects);
}

template<typename MarkingCategory>
void
checkMarkingSpatialRelation(
    CheckContext* context,
    const std::unordered_map<TId, const RoadSurface*>& roadSurfacesById,
    const std::string& descriptionInfix)
{
    context->objects<MarkingCategory>().visit([&](
        const typename MarkingCategory::TObject* marking)
    {
        if (marking->roadSurface()) {
            if (!isMajorIntersection(
                    roadSurfacesById.at(marking->roadSurface())->geom(),
                    marking->geom()))
            {
                context->critical(
                    "road-marking-" + descriptionInfix + "-outside-surface",
                    geomForReport(marking),
                    {marking->id()});
            }
            return;
        }

        auto hasRoadSurface = false;
        size_t zLeveledSurfacesCount = 0;

        for (const auto& [roadSurfaceId, roadSurface] : roadSurfacesById) {
            if (geolib3::spatialRelation(
                    roadSurface->geom(), marking->geom(), geolib3::Intersects))
            {
                hasRoadSurface = true;
                if (roadSurface->fromZlevel() || roadSurface->toZlevel()) {
                    ++zLeveledSurfacesCount;
                }
            }
        }

        if (!hasRoadSurface) {
            context->critical(
                "road-marking-" + descriptionInfix + "-no-surface",
                geomForReport(marking),
                {marking->id()});
        }
        if (zLeveledSurfacesCount > 1) {
            context->critical(
                "road-marking-" + descriptionInfix + "-ambiguous-surface",
                geomForReport(marking),
                {marking->id()});
        }
    });
}

} // namespace

VALIDATOR_SIMPLE_CHECK( road_marking_spatial_relation,
    ROAD_SURFACE,
    ROAD_MARKING_POLYGONAL,
    ROAD_MARKING_LINEAR,
    ROAD_MARKING_POINT_LANE_DIRECTION,
    ROAD_MARKING_POINT_SYMBOL,
    ROAD_MARKING_POINT_ROAD_SIGN,
    ROAD_MARKING_POINT_ROAD_SIGN_SPEED_LIMIT,
    ROAD_MARKING_POINT_TEXT )
{
    std::unordered_map<TId, const RoadSurface*> roadSurfacesById;
    context->objects<ROAD_SURFACE>().visit([&](const RoadSurface* roadSurface)
    {
        roadSurfacesById[roadSurface->id()] = roadSurface;
    });

    checkMarkingSpatialRelation<ROAD_MARKING_POLYGONAL>(context, roadSurfacesById, "polygonal");
    checkMarkingSpatialRelation<ROAD_MARKING_LINEAR>(context, roadSurfacesById, "linear");

    checkMarkingSpatialRelation<ROAD_MARKING_POINT_LANE_DIRECTION>(context, roadSurfacesById, "point");
    checkMarkingSpatialRelation<ROAD_MARKING_POINT_SYMBOL>(context, roadSurfacesById, "point");
    checkMarkingSpatialRelation<ROAD_MARKING_POINT_ROAD_SIGN>(context, roadSurfacesById, "point");
    checkMarkingSpatialRelation<ROAD_MARKING_POINT_ROAD_SIGN_SPEED_LIMIT>(context, roadSurfacesById, "point");
    checkMarkingSpatialRelation<ROAD_MARKING_POINT_TEXT>(context, roadSurfacesById, "point");
}

} // maps::wiki::validator::checks
