#include "module.h"
#include "../utils/misc.h"

#include <yandex/maps/wiki/validator/check.h>
#include <yandex/maps/wiki/validator/categories.h>

#include <maps/libs/geolib/include/convex_hull.h>
#include <maps/libs/geolib/include/intersection.h>

using maps::wiki::validator::categories::HYDRO_FC;
using maps::wiki::validator::categories::HYDRO_FC_EL;
using maps::wiki::validator::categories::RELIEF;
using maps::wiki::validator::categories::RELIEF_FC;
using maps::wiki::validator::categories::RELIEF_EL;

namespace maps {
namespace wiki {
namespace validator {
namespace checks {

namespace {
const size_t VISIT_BATCH_SIZE = 10000;
} // namespace

VALIDATOR_SIMPLE_CHECK( hydro_relief_intersections, HYDRO_FC, HYDRO_FC_EL, RELIEF, RELIEF_FC, RELIEF_EL )
{
    std::unordered_set<TId> exteriorHydroFcElIds;
    std::unordered_set<TId> namedReliefElIds;

    context->objects<HYDRO_FC>().visit(
        [&](const Face* hydroFc) {
            if (!hydroFc->isInterior()) {
                exteriorHydroFcElIds.insert(
                    hydroFc->edges().begin(),
                    hydroFc->edges().end());
            }
        }
    );

    context->objects<RELIEF>().visit(
        [&](const ContourFeature* relief) {
            if (relief->names().empty()) {
                return;
            }
            for (auto reliefFcId : relief->faces()) {
                if (context->objects<RELIEF_FC>().loaded(reliefFcId)) {
                    auto reliefFc = context->objects<RELIEF_FC>().byId(reliefFcId);
                    for (auto reliefElId : reliefFc->edges()) {
                        namedReliefElIds.insert(reliefElId);
                    }
                }
            }
        }
    );

    context->objects<RELIEF_EL>().batchVisit(
        [&](const Edge* reliefEl) {
            const auto& hydroFcEls = context->objects<HYDRO_FC_EL>().byBbox(
                reliefEl->geom().boundingBox());

            for (const auto& hydroFcEl : hydroFcEls) {
                if (!exteriorHydroFcElIds.count(hydroFcEl->id())) {
                    continue;
                }

                const auto intersection = geolib3::intersection(
                    hydroFcEl->geom(), reliefEl->geom());

                if (!intersection.empty()) {
                    context->report(
                        namedReliefElIds.count(reliefEl->id()) ? Severity::Critical : Severity::Error,
                        "hydro-relief-intersection",
                        geolib3::bufferedConvexHull(intersection.front().points(), utils::BUFFER_DISTANCE),
                        { hydroFcEl->id(), reliefEl->id() });
                }
            }
        }, VISIT_BATCH_SIZE);
}

} // namespace checks
} // namespace validator
} // namespace wiki
} // namespace maps
