#include "module.h"
#include "poi_categories.h"
#include "../utils/geom.h"

#include <maps/wikimap/mapspro/libs/poi_conflicts/include/poi_conflicts.h>
#include <yandex/maps/wiki/validator/check.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>


#include <unordered_map>

namespace maps::wiki::validator::checks {

#define OTHER_CATEGORIES \
    categories::ADDR, \
    categories::TRANSPORT_AIRPORT, \
    categories::TRANSPORT_AIRPORT_TERMINAL, \
    categories::TRANSPORT_HELICOPTER, \
    categories::TRANSPORT_METRO_STATION, \
    categories::TRANSPORT_METRO_EXIT, \
    categories::TRANSPORT_RAILWAY_STATION, \
    categories::TRANSPORT_STOP, \
    categories::TRANSPORT_TERMINAL, \
    categories::VEGETATION_CNT


namespace {
const size_t CRITICAL_CONFLICT_MIN_ZOOM = 20;

static const poi_conflicts::PoiConflicts g_poiConflicts;

struct PoiGeomIndexKey
{
    PoiGeomIndexKey(const geolib3::Point2& pt)
        : position(pt)
    {
        size = 2 * g_poiConflicts.maxConflictDistanceMercator();
    }
    geolib3::BoundingBox boundingBox() const
    {
        return geolib3::BoundingBox(
            position,
            size,
            size);
    }

    geolib3::Point2 position;
    double size;
};

template<class Category>
bool indexGeoproductsCategoryCoordinates(
    CheckContext* context,
    std::unordered_map<TId, geolib3::Point2>& geoProductPoiIdToGeom)
{
    context->objects<Category>().visit([&](const Poi* poi) {
        if (poi->isGeoproduct()) {
            geoProductPoiIdToGeom.emplace(poi->id(), poi->geom());
        }
    });
    return true;
}

template<class... Categories>
std::unordered_map<TId, geolib3::Point2>
indexGeoproductsCategoriesCoordinates(
    CheckContext* context)
{
    std::unordered_map<TId, geolib3::Point2> geoProductPoiIdToGeom;
    auto performed = {
        indexGeoproductsCategoryCoordinates<Categories>(context, geoProductPoiIdToGeom)...
    };
    (void)performed;
    return geoProductPoiIdToGeom;
}

struct ConflictCandiatesSearchIndex
{
    ConflictCandiatesSearchIndex(CheckContext* context)
    {
        indexConflictCategoriesCandidates<POI_CATEGORIES, OTHER_CATEGORIES>(
            context);
    }

    template<class Category>
    bool indexConflictCategoryCandidates(
        CheckContext* context)
    {
        context->objects<Category>().visit(
            [&](const typename Category::TObject* poi) {
                data.emplace_back(poi->geom());
                index.insert(
                    &data.back(),
                    poi->id());
            });
        return true;
    }

    template<class... Categories>
    void
    indexConflictCategoriesCandidates(CheckContext* context)
    {
        auto performed = {
                indexConflictCategoryCandidates<Categories>(
                    context)...
        };
        index.build();
        (void)performed;
    }

    std::list<PoiGeomIndexKey> data;
    geolib3::StaticGeometrySearcher<PoiGeomIndexKey, TId> index;
};

std::optional<Severity>
conflictSeverity(
    const geolib3::Point2& geoProductPoiPosition,
    const geolib3::Point2& positionToCheck,
    bool otherIsGeoproduct)
{
    auto conflictZoom = g_poiConflicts.conflictZoom(
        geolib3::distance(geoProductPoiPosition, positionToCheck));
    if (!conflictZoom) {
        return std::nullopt;
    }
    return conflictZoom >= CRITICAL_CONFLICT_MIN_ZOOM || otherIsGeoproduct
        ? Severity::Critical
        : Severity::Error;
}

} // namespace

VALIDATOR_SIMPLE_CHECK(poi_conflicts, POI_CATEGORIES, OTHER_CATEGORIES)
{
    const auto geoProductPoiIdToGeom =
        indexGeoproductsCategoriesCoordinates<POI_CATEGORIES>(
            context);
    const ConflictCandiatesSearchIndex conflictCandiatesSearchIndex(context);
    for (const auto& [id, geom] : geoProductPoiIdToGeom) {
        auto possibleConflictObjects =
            conflictCandiatesSearchIndex.index.find(
                PoiGeomIndexKey(geom).boundingBox());
        for (auto it = possibleConflictObjects.first;
            it != possibleConflictObjects.second; ++it)
        {
            const auto& findingId = it->value();
            if (findingId == id) {
                continue;
            }
            auto severity = conflictSeverity(
                geom,
                it->geometry().position,
                geoProductPoiIdToGeom.count(findingId));
            if (severity) {
                context->report(
                    *severity,
                    "poi-visibility-conflict",
                    geom,
                    {id, findingId});
            }
        }
    }
}
} // namespace maps::wiki::validator::checks
