#include "module.h"
#include "../utils/geom.h"

#include <yandex/maps/wiki/validator/categories.h>
#include <yandex/maps/wiki/validator/check.h>
#include <maps/libs/geolib/include/vector.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>

namespace maps {
namespace wiki {
namespace validator {
namespace checks {

namespace {

const double MIN_DISTANCE_TO_POI_ENTRANCE = 1; // meters
const double MAX_DISTANCE_TO_BLD = 0.5; // meters

using PointGeomSearcher = geolib3::StaticGeometrySearcher<geolib3::Point2, TId>;
using PolygonGeomSearcher = geolib3::StaticGeometrySearcher<geolib3::Polygon2, TId>;

bool isEntranceCloseToBld(
    const geolib3::Point2& entranceGeom,
    const PolygonGeomSearcher& bldSearcher)
{
    auto distanceRatio = utils::mercatorDistanceRatio(entranceGeom);

    auto searchResult = bldSearcher.find(
        geolib3::BoundingBox(entranceGeom,
            2 * MAX_DISTANCE_TO_BLD / distanceRatio,
            2 * MAX_DISTANCE_TO_BLD / distanceRatio));

    for (auto it = searchResult.first; it != searchResult.second; ++it) {
        if (geolib3::distance(entranceGeom, it->geometry()) * distanceRatio <
                MAX_DISTANCE_TO_BLD)
        {
            return true;
        }
    }
    return false;;
}

} // namespace

using categories::POI_ENTRANCE;
using categories::BLD;

VALIDATOR_SIMPLE_CHECK( poi_entrance_geometry, POI_ENTRANCE, BLD )
{
    PointGeomSearcher entranceSearcher;
    PolygonGeomSearcher bldSearcher;

    context->objects<POI_ENTRANCE>().visit([&](const PoiEntrance* entrance) {
        entranceSearcher.insert(&entrance->geom(), entrance->id());
    });
    context->objects<BLD>().visit([&](const Building* bld) {
        bldSearcher.insert(&bld->geom(), bld->id());
    });

    entranceSearcher.build();
    bldSearcher.build();

    std::set<TId> reportedCloseEntranceIds;

    context->objects<POI_ENTRANCE>().visit([&](const PoiEntrance* entrance) {
        if (!isEntranceCloseToBld(entrance->geom(), bldSearcher)) {
            context->error(
                "poi-entrance-too-far-from-bld",
                entrance->geom(),
                {entrance->id()});
        }

        if (reportedCloseEntranceIds.count(entrance->id())) {
            return;
        }

        auto distanceRatio = utils::mercatorDistanceRatio(entrance->geom());
        auto searchResult = entranceSearcher.find(
            geolib3::BoundingBox(entrance->geom(),
                2 * MIN_DISTANCE_TO_POI_ENTRANCE / distanceRatio,
                2 * MIN_DISTANCE_TO_POI_ENTRANCE / distanceRatio));
        std::vector<TId> closeEntranceIds;
        for (auto it = searchResult.first; it != searchResult.second; ++it) {
            closeEntranceIds.push_back(it->value());
        }

        reportedCloseEntranceIds.insert(closeEntranceIds.begin(), closeEntranceIds.end());
        if (closeEntranceIds.size() > 1) {
            context->error(
                "poi-entrance-too-close",
                entrance->geom(),
                closeEntranceIds);
        }
    });
}

} // namespace checks
} // namespace validator
} // namespace wiki
} // namespace maps
