#include "module.h"
#include <yandex/maps/wiki/validator/check.h>

#include "poi_categories.h"
#include "indoor_poi_categories.h"
#include "../utils/names.h"
#include "../utils/name_data.h"

#include <functional>

namespace maps {
namespace wiki {
namespace validator {

namespace utils {
namespace {

template<>
struct GeomForReport<Poi> {
    static geolib3::Point2 geom(const Poi* poi) {
        return poi->geom();
    }
};

} // namespace utils
} // namespace

namespace checks {

using namespace categories;

namespace {

const icu::RegexPattern ILLEGAL_SYMBOL_REGEX =
    utils::compilePattern("[\\[\\]{}()<>\"«»“”„]");

constexpr size_t MAX_NAME_LEN = 55;
constexpr size_t MAX_ENTRANCE_NAME_LEN = 9;

template<class CategoryName, class Category>
bool performCheck(CheckContext* context)
{
    context->objects<Category>().visit([&](const Poi* poi)
    {
        utils::runBasicNamesCheck<CategoryName>(poi, context, Severity::Critical);
        utils::runCapsNameCheck<CategoryName>(poi, context);

        utils::visitRenderedNames<CategoryName>(
            poi,
            context,
            [](const Poi* poi, CheckContext* ctx, const std::string& name) {
                if (utils::countCodePoints(name) > MAX_NAME_LEN) {
                    ctx->error(
                        "max-name-length-exceeded", poi->geom(), {poi->id()});
                }
                if (utils::matchesPattern(name, ILLEGAL_SYMBOL_REGEX)) {
                    ctx->error(
                        "illegal-symbol-in-poi-name", poi->geom(), {poi->id()});
                }
            }
        );

    });
    return true;
}

template<class CategoryName, class... Categories>
void performChecks(CheckContext* context)
{
    auto performed = {performCheck<CategoryName, Categories>(context)...};
    (void)performed;
}

} // namespace

VALIDATOR_SIMPLE_CHECK( poi_naming, POI_NM, POI_CATEGORIES )
{
    performChecks<POI_NM, POI_CATEGORIES>(context);
}

VALIDATOR_SIMPLE_CHECK( indoor_poi_naming, INDOOR_POI_NM, INDOOR_POI_CATEGORIES )
{
    performChecks<INDOOR_POI_NM, INDOOR_POI_CATEGORIES>(context);
}

VALIDATOR_SIMPLE_CHECK( poi_entrance_naming, POI_NM, POI_ENTRANCE )
{
    context->objects<POI_ENTRANCE>().visit([&](const PoiEntrance* entrance) {
        for (const NameRelation& rel : entrance->names()) {
            const Name* name = context->objects<POI_NM>().byId(rel.id);
            if (utils::countCodePoints(name->name()) > MAX_ENTRANCE_NAME_LEN) {
                context->error(
                    "max-entrance-name-length-exceeded",
                    entrance->geom(),
                    { entrance->id() });
                break;
            }
        }
    });
}

} // namespace checks
} // namespace validator
} // namespace wiki
} // namespace maps
