#include "module.h"

#include <yandex/maps/wiki/validator/check.h>
#include <yandex/maps/wiki/validator/categories.h>

#include "../utils/find_gaps_representatives.h"
#include "../utils/geom.h"
#include "../utils/misc.h"
#include "../utils/object_elements_within_aoi.h"

#include <maps/libs/geolib/include/closest_point.h>
#include <maps/libs/geolib/include/convex_hull.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polygon.h>

#include <vector>

namespace maps {
namespace wiki {
namespace validator {
namespace checks {

namespace {

static const double MIN_GAP_SIZE = 2e3; // meters

const size_t VISIT_BATCH_SIZE = 5000;

geolib3::Polygon2 geomForReport(
        const geolib3::Polyline2& a,
        const geolib3::Polyline2& b)
{
    geolib3::TwoPoints points(geolib3::closestPoints(a, b));

    return geolib3::bufferedConvexHull(
        {points.first, points.second},
        utils::BUFFER_DISTANCE);
}

} // namespace

using categories::RD;
using categories::RD_EL;

VALIDATOR_SIMPLE_CHECK( road_gaps, RD, RD_EL )
{
    auto viewRdEl = context->objects<RD_EL>();

    context->objects<RD>().batchVisit(
            [&](const Road* road)
    {
        if (!utils::objectElementsWithinAoi<RD>(context, road)) {
            return;
        }

        const auto& elements = road->elements();

        geolib3::PolylinesVector geoms;
        geoms.reserve(elements.size());
        for (TId elementId : elements) {
            geoms.push_back(viewRdEl.byId(elementId)->geom());
        }

        for (const auto& gapRepresentatives
                : utils::findGapsRepresentatives(geoms, MIN_GAP_SIZE)) {
            context->error(
                "road-gap-too-big",
                geomForReport(
                    geoms[gapRepresentatives.first],
                    geoms[gapRepresentatives.second]),
                { road->id(),
                  elements[gapRepresentatives.first],
                  elements[gapRepresentatives.second] });
        }
    }, VISIT_BATCH_SIZE);
}

} // namespace checks
} // namespace validator
} // namespace wiki
} // namespace maps
