#include "module.h"
#include <yandex/maps/wiki/validator/check.h>
#include <yandex/maps/wiki/validator/categories.h>

using maps::wiki::validator::categories::REGION;
using maps::wiki::validator::categories::AD;
using maps::wiki::validator::categories::AD_NEUTRAL;

namespace maps {
namespace wiki {
namespace validator {
namespace checks {

VALIDATOR_SIMPLE_CHECK( region_relations, REGION, AD, AD_NEUTRAL )
{
    std::map<TId, std::set<TId>> adToRegions;
    std::map<TId, std::set<TId>> adNeutralToRegions;

    context->objects<REGION>().visit([&](const Region* region) {
        for (auto admUnitId : region->admUnits()) {
            const auto* admUnit = context->objects<AD>().byId(admUnitId);
            if (admUnit->levelKind() == AdmUnit::LevelKind::Country) {
                adToRegions[admUnitId].insert(region->id());
            } else {
                context->critical(
                    "region-has-ad-with-wrong-level-kind",
                    region->geom(),
                    { region->id(), admUnitId });
            }
        }

        for (auto adNeutralId : region->adNeutrals()) {
            adNeutralToRegions[adNeutralId].insert(region->id());
        }
    });

    auto checkCardinality = [&context](
            const std::map<TId, std::set<TId>>& elementToRegion,
            const std::string& prefix) {
        for (const auto& pair : elementToRegion) {
            auto elementId = pair.first;
            const auto& regionIds = pair.second;
            if (regionIds.size() > 1) {
                std::vector<TId> reportOids;
                reportOids.push_back(elementId);
                reportOids.insert(reportOids.end(), regionIds.begin(), regionIds.end());

                context->critical(
                    prefix + "-is-attached-to-multiple-regions",
                    boost::none,
                    reportOids);
            }
        }
    };

    checkCardinality(adToRegions, "ad");
    checkCardinality(adNeutralToRegions, "ad-neutral");

    context->objects<AD>().visit([&](const AdmUnit* admUnit) {
        if (admUnit->levelKind() == AdmUnit::LevelKind::Country) {
            if (!adToRegions.count(admUnit->id())) {
                context->critical(
                    "ad-is-not-attached-to-region",
                    boost::none,
                    { admUnit->id() });
            }
        }
    });

    context->objects<AD_NEUTRAL>().visit([&](const ContourFeature* adNeutral) {
        if (!adNeutralToRegions.count(adNeutral->id())) {
            context->critical(
                "ad-neutral-is-not-attached-to-region",
                boost::none,
                { adNeutral->id() });
        }
    });
}

} // namespace checks
} // namespace validator
} // namespace wiki
} // namespace maps
