#include "module.h"
#include "../utils/road_utils.h"

#include <yandex/maps/wiki/validator/check.h>
#include <yandex/maps/wiki/validator/categories.h>
#include <yandex/maps/wiki/common/misc.h>

#include <map>
#include <vector>

namespace maps::wiki::validator::checks {

using categories::RD_EL;
using categories::RD_JC;
using categories::COND;

namespace {

bool
reportIfDeadEnd(
    CheckContext* context,
    const utils::RdEls& rdEls,
    const Junction* rdJc,
    const utils::RdElContinuationsByJcId& prohibitedContinuations)
{
    if (rdEls.size() == 1 &&
        (*rdEls.begin())->direction() == RoadElement::Direction::Both)
    {
        return false;
    }
    if (rdEls.size() == 2 &&
        rdEls[0]->fc() == 8 &&
        rdEls[1]->fc() == 8 &&
        prohibitedContinuations.count(rdJc->id()) &&
        prohibitedContinuations.find(rdJc->id())->second.size() == 2)
    {
        return false;
    }

    for (const auto& rdEl : rdEls) {
        bool isDeadEndForTo = true;
        if (!utils::isRdElCanBeTo(rdEl, rdJc)) {
            isDeadEndForTo = false;
        } else {
            for (const auto& continuation : rdEls) {
                if (continuation == rdEl) {
                    continue;
                }
                if (utils::canDrive(
                        rdEl,
                        rdJc,
                        continuation,
                        prohibitedContinuations)) {
                    isDeadEndForTo = false;
                    break;
                }
            }
        }

        bool isDeadEndForFrom = true;
        if (!utils::isRdElCanBeFrom(rdEl, rdJc)) {
            isDeadEndForFrom = false;
        } else {
            for (const auto& continuation : rdEls) {
                if (continuation == rdEl) {
                    continue;
                }
                if (utils::canDrive(
                        continuation,
                        rdJc,
                        rdEl,
                        prohibitedContinuations)) {
                    isDeadEndForFrom = false;
                    break;
                }
            }
        }

        if (isDeadEndForTo || isDeadEndForFrom) {
            context->report(
                rdEl->fc() <= 6 ?
                    Severity::Critical :
                    Severity::Error,
                "dead-end",
                rdJc->geom(),
                {rdJc->id(), rdEl->id()});
            return true;
        }
    }
    return false;
}

bool
reportIfNoContinuation(
    CheckContext* context,
    const utils::RdEls& rdEls,
    const Junction* rdJc)
{
    int minFc = RoadElement::MAX_FC + 1;
    size_t minFcCount = 0;
    TId rdElIdWithMinFc = 0;

    for (const auto& rdEl : rdEls) {
        auto fc = rdEl->fc();
        if (fc < minFc) {
            minFc = fc;
            minFcCount = 1;
            rdElIdWithMinFc = rdEl->id();
        } else if (fc == minFc) {
            ++minFcCount;
        }
    }

    if (minFc <= 6 && minFcCount == 1) {
        context->report(
            minFc <= 4 ?
                Severity::Critical :
                Severity::Error,
            "no-continuation-high-fc",
            rdJc->geom(),
            {rdJc->id(), rdElIdWithMinFc});
        return true;
    }
    return false;
}

bool
reportIfDualCarriagewayDeadEnd(
    CheckContext* context,
    const utils::RdEls& rdEls,
    const Junction* rdJc)
{
    size_t dualCarriagewayCount = 0;
    size_t dualCarriagewayFromCount = 0;
    size_t dualCarriagewayToCount = 0;

    for (const auto& rdEl : rdEls) {
        if (rdEl->fow() == common::FOW::TwoCarriageway) {
            ++dualCarriagewayCount;
            if (utils::isRdElCanBeFrom(rdEl, rdJc)) {
                ++dualCarriagewayFromCount;
            }
            if (utils::isRdElCanBeTo(rdEl, rdJc)) {
                ++dualCarriagewayToCount;
            }
        }
    }

    if (rdEls.size() == 4 && common::isIn(dualCarriagewayCount, {2, 4}) &&
        dualCarriagewayFromCount != dualCarriagewayToCount) {
        context->warning(
            "dual-carriageway-dead-end",
            rdJc->geom(),
            {rdJc->id()});
        return true;
    }
    return false;
}

} // namespace

VALIDATOR_SIMPLE_CHECK( dead_ends, RD_EL, RD_JC, COND )
{
    auto prohibitedContinuations = utils::findProhibitedContinuations(
        context, common::AccessId::Car);

    context->objects<RD_JC>().visit(
        [&](const Junction* rdJc) {
            auto rdElsByZLevel = utils::groupRdElByZLevForJc(
                context, rdJc, common::AccessId::Car);

            for (const auto& pair : rdElsByZLevel) {
                const auto& rdEls = pair.second;
                if (reportIfDeadEnd(context, rdEls, rdJc, prohibitedContinuations)) {
                    break;
                }
            }
            for (const auto& pair : rdElsByZLevel) {
                const auto& rdEls = pair.second;
                if (reportIfNoContinuation(context, rdEls, rdJc)) {
                    break;
                }
            }
            for (const auto& pair : rdElsByZLevel) {
                const auto& rdEls = pair.second;
                if (reportIfDualCarriagewayDeadEnd(context, rdEls, rdJc)) {
                    break;
                }
            }
        });
}

} // namespace maps::wiki::validator::checks
