#include "module.h"
#include <yandex/maps/wiki/validator/check.h>
#include <yandex/maps/wiki/validator/categories.h>

#include "../utils/misc.h"
#include "../utils/road_utils.h"

#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/geolib/include/conversion.h>

namespace gl = maps::geolib3;

namespace maps {
namespace wiki {
namespace validator {
namespace checks {

using categories::RD_EL;
using categories::RD_JC;

namespace {

const std::string MSG_WRONG_ZLEV = "wrong-zlev-in-incident-rd_els";
const std::string MSG_NO_CONTINUATION_ON_CURRENT_ZLEV =
    "no-continuation-on-current-zlev";
const std::string MSG_WRONG_CROSSING_ZLEV = "wrong-highway-crossing-zlev";

const double MAX_PAIR_TURN_ANGLE = M_PI / 3; // radians

struct ElementInfo
{
    const RoadElement* element;
    common::AccessId accessId;
    int zlevel;
    gl::Direction2 incidenceBearing;
};

gl::Direction2 incidenceBearing(
    const RoadElement* element,
    const Junction* junction)
{
    size_t iCand = (junction->id() == element->startJunction() ?
                    1 : element->geom().pointsNumber() - 2);
    while (iCand > 0 && iCand + 1 < element->geom().pointsNumber() &&
           gl::distance(junction->geom(), element->geom().points()[iCand]) < utils::EPS) {
        iCand += (junction->id() == element->startJunction() ? 1 : -1);
    }
    return gl::Direction2(gl::Segment2(
        element->geom().points()[iCand], junction->geom()));
}

} // namespace

VALIDATOR_SIMPLE_CHECK( zlevels, RD_EL, RD_JC )
{
    context->objects<RD_JC>().visit(
        [&](const Junction* junction)
    {
        std::vector<ElementInfo> elementsInfo;
        for (TId id : junction->inElements()) {
            if (!context->objects<RD_EL>().loaded(id)) {
                continue;
            }
            auto elem = context->objects<RD_EL>().byId(id);
            elementsInfo.push_back(ElementInfo{
                elem,
                elem->accessId(),
                elem->toZlevel(),
                incidenceBearing(elem, junction)});
        }
        for (TId id : junction->outElements()) {
            if (!context->objects<RD_EL>().loaded(id)) {
                continue;
            }
            auto elem = context->objects<RD_EL>().byId(id);
            elementsInfo.push_back(ElementInfo{
                elem,
                elem->accessId(),
                elem->fromZlevel(),
                incidenceBearing(elem, junction)});
        }

        auto allHaveSameZLevel = std::all_of(elementsInfo.begin(), elementsInfo.end(),
            [&elementsInfo](const ElementInfo& ei) {
                return ei.zlevel == elementsInfo[0].zlevel;
            });
        if (allHaveSameZLevel) {
            auto allVehicleHaveSameZLevel = std::all_of(elementsInfo.begin(), elementsInfo.end(),
                [&elementsInfo](const ElementInfo& ei) {
                    return (ei.accessId | utils::notVehicle) != utils::notVehicle &&
                        ei.zlevel == elementsInfo[0].zlevel;
                });
            if (allVehicleHaveSameZLevel) {
                if (elementsInfo.size() >= 4 && elementsInfo[0].zlevel != 0) {
                    context->critical(MSG_WRONG_CROSSING_ZLEV, junction->geom(), {junction->id()});
                }
            }
            return;
        }

        if (elementsInfo.size() == 2 &&
            elementsInfo[0].zlevel != elementsInfo[1].zlevel) {
            context->critical(
                MSG_WRONG_ZLEV,
                junction->geom(),
                {junction->id(),
                 elementsInfo[0].element->id(), elementsInfo[1].element->id()});
            return;
        }

        for (const auto& elInfo : elementsInfo) {
            auto isElemPair = [&elInfo](const ElementInfo& other)
                {
                    return elInfo.zlevel == other.zlevel &&
                        gl::angle(elInfo.incidenceBearing, other.incidenceBearing)
                            > M_PI - MAX_PAIR_TURN_ANGLE;
                };

            if (std::none_of(elementsInfo.begin(), elementsInfo.end(),
                             isElemPair)) {
                context->critical(MSG_NO_CONTINUATION_ON_CURRENT_ZLEV, junction->geom(),
                    {elInfo.element->id(), junction->id()});
                return;
            }
        }
    });
}

} // namespace checks
} // namespace validator
} // namespace wiki
} // namespace maps
